import assign from 'lodash/assign';
import { ACTION_CLIP_RECOMMENDATIONS_LOADED, CATEGORY_ORDER } from 'actions/clip-recommendations';
import { addIfNotPresent } from 'ui/player-types/clips/utils/state-helper';

const DEFAULT_CLIP_RECOMMENDATIONS = Object.freeze({
    byCategory: {}, // category -> [] list of clipSlugs
    allCategories: [], // list of all categories
    hasFetched: {}, // recommendation request -> boolean representing if that fetch has gone through
});

/**
 * Store reducer for Clip Recommendations state.
 *
 * @param {Object} clipRecommendations
 * @param {Action} action
 * @return {Object}
 */
export function clipRecommendations(currentState = DEFAULT_CLIP_RECOMMENDATIONS, action) {
    switch (action.type) {
    case ACTION_CLIP_RECOMMENDATIONS_LOADED: {
        const newState = assign({}, currentState);

        Object.keys(action.recommendationSets).forEach(clipsSetKey => {
            newState.byCategory[clipsSetKey] = action.recommendationSets[clipsSetKey].map(clip => clip.slug);
            newState.allCategories = addIfNotPresent(newState.allCategories, clipsSetKey);
        });

        newState.hasFetched[action.recommendationId] = true;

        return newState;
    }
    default:
        return currentState;
    }
}

// Selects first category with any clips by category preference
export const getFirstValidCategory = (state, recommendationId) => {
    const { clipRecommendations } = state;

    if (!clipRecommendations.hasFetched[recommendationId]) {
        return [];
    }

    for (const idx in CATEGORY_ORDER) {
        const category = CATEGORY_ORDER[idx];

        if (clipRecommendations.byCategory[category]) {
            return category;
        }
    }
};

export const getRecommendedClipsForSlug = (state, recommendationId) => {
    const { clipRecommendations, clipsData } = state;

    if (!clipRecommendations.hasFetched[recommendationId]) {
        return [];
    }

    return CATEGORY_ORDER.reduce((clipsInPriorityOrder, category) => {
        const clipSlugsForCateogry = clipRecommendations.byCategory[category];
        const clipForCateogry = clipSlugsForCateogry.map(clipSlug => clipsData.bySlug[clipSlug]);
        return clipsInPriorityOrder.concat(clipForCateogry);
    }, []);
};
