import assign from 'lodash/assign';
import { ACTION_CLIP_GENERATOR_LOADED } from 'actions/clips';
import { BACKEND_HLS } from 'backend/hls';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { LOGGED_IN } from 'state/user';
import { ONLINE_STATUS } from 'state/online-status';
import {
    PLAYER_HIGHLIGHTER,
    PLAYER_IMDB,
    PLAYER_AMAZON_VSE,
    PLAYER_AMAZON_LIVE,
    PLAYER_CLIPS_VIEWING,
    PLAYER_CLIPS_EMBED,
} from 'util/player-type';
import includes from 'lodash/includes';

export const NO_CLIPS_PLAYER_TYPES = Object.freeze([
    PLAYER_IMDB,
    PLAYER_AMAZON_VSE,
    PLAYER_HIGHLIGHTER,
    PLAYER_AMAZON_LIVE,
    PLAYER_CLIPS_VIEWING,
    PLAYER_CLIPS_EMBED,
]);

export const NO_CLIPS_CHANNELS = Object.freeze([
    // Forever clipless
    'misterrogers',
    'contv',
    'failarmy',

    // For Football, removal timing not crucial - but anytime after 12/08/2017
    'primevideo',
]);

const DEFAULT_GENERATOR = Object.freeze({
    recordClip() {
        // eslint-disable-next-line no-console
        console.error('Clip Generator not available yet');
    },
});

export const DEFAULT_CLIPS_STATE = Object.freeze({
    generator: DEFAULT_GENERATOR,
});

/**
 * Store reducer for clips.
 *
 * @param {Object} clipsState
 * @param {Action} action
 * @return {Object}
 */
export function clips(clipsState = DEFAULT_CLIPS_STATE, action) {
    switch (action.type) {
    case ACTION_CLIP_GENERATOR_LOADED:
        return assign({}, clipsState, {
            generator: action.generator,
        });
    default:
        return clipsState;
    }
}

export function isValidChannelState({ streamMetadata }) {
    const hasBroadcastId = Boolean(streamMetadata.broadcastID);
    const isClipsSupportedChannel = !includes(NO_CLIPS_CHANNELS, streamMetadata.channel.name);

    return hasBroadcastId && isClipsSupportedChannel;
}

export function isValidPlaybackState({ playback, stream }) {
    const hasPlayed = playback.hasPlayed;
    const isPausedLiveStream = stream.contentType === CONTENT_MODE_LIVE && playback.paused;

    return hasPlayed && !isPausedLiveStream;
}

export function isValidStreamState({ stream, onlineStatus, env, backendInfo }) {
    const isOfflineLiveStream = stream.contentType === CONTENT_MODE_LIVE && onlineStatus !== ONLINE_STATUS;
    const isInvalidStreamType = !includes([CONTENT_MODE_LIVE, CONTENT_MODE_VOD], stream.contentType);
    const isInvalidPlayerType = includes(NO_CLIPS_PLAYER_TYPES, env.playerType);
    const isUnsupportedBackend = backendInfo.playerBackendType === BACKEND_HLS;

    return !(
        isOfflineLiveStream ||
        isInvalidStreamType ||
        isUnsupportedBackend ||
        isInvalidPlayerType
    );
}

export const userCanClipSelector = state => {
    const isLoggedIn = state.user.loggedInStatus === LOGGED_IN;

    return (
        isLoggedIn &&
        isValidChannelState(state) &&
        isValidPlaybackState(state) &&
        isValidStreamState(state)
    );
};
