import assign from 'lodash/assign';
import bowser from 'bowser';
import { parseUri } from '../util/parseuri';
import { PLAYER_POPOUT } from '../util/player-type';
import { ACTION_SET_ENVIRONMENT, ACTION_SET_PLAYER_TYPE, ACTION_SET_TWITCH_EVERYWHERE_PARAMS } from '../actions/env';
import { ACTION_SET_WINDOW } from '../actions/window';
import { ALL_PLAYER_TYPES } from 'util/player-type';
import includes from 'lodash/includes';
import toLower from 'lodash/toLower';
import endsWith from 'lodash/endsWith';

export const PLATFORM_WEB = 'web';
export const PLATFORM_MOBILE_WEB = 'mobile_web';

export const DEFAULT_ENVIRONMENT = Object.freeze({
    deviceId: '',
    playerType: PLAYER_POPOUT,
    platform: PLATFORM_WEB,
    isMobileLocation: false,
});

const MOBILE_HOSTNAME = 'm.twitch.tv';
const HTTPS_PROTOCOL = 'https';

/**
 * Store reducer for player environment.
 *
 * @param {?Object} environment
 * @param {Action} action
 * @return {Object}
 */
export function env(environment = DEFAULT_ENVIRONMENT, action) {
    switch (action.type) {
    case ACTION_SET_ENVIRONMENT:
        return assign(
            {},
            environment,
            {
                deviceId: action.deviceId,
            }
        );
    // eslint-disable-next-line no-case-declarations
    case ACTION_SET_PLAYER_TYPE:
        const playerType = includes(ALL_PLAYER_TYPES, action.playerType) ? action.playerType : environment.playerType;
        return assign(
            {},
            environment,
            {
                playerType,
            }
        );
    case ACTION_SET_TWITCH_EVERYWHERE_PARAMS:
        return assign(
            {},
            environment,
            {
                twitchEverywhere: action.params,
            }
        );
    case ACTION_SET_WINDOW:
        return assign(
            {},
            environment,
            {
                platform: determinePlatform((action.window.navigator || {}).userAgent),
                isMobileLocation: isMobileLocation(action.window),
            }
        );
    default:
        return environment;
    }
}

/**
 * Mobile web platform detector.
 *
 * @param {String} userAgent
 * @return {Boolean}
 */
function determinePlatform(userAgent = '') {
    const uaFingerprint = bowser._detect(userAgent);
    return uaFingerprint.mobile || uaFingerprint.tablet ? PLATFORM_MOBILE_WEB : PLATFORM_WEB;
}

/**
  * Determine if the current location is part of the mobile upsell funnel.
  *
  * @param {Window} window
  * @return {Boolean}
 **/
function isMobileLocation(window) {
    const referrer = parseUri(window.document.referrer);
    let isEmbedded;
    try {
        isEmbedded = window !== window.top;
    } catch (e) {
        // You won't get the security policy exception if you are not embedded
        // because, by definition, it isn't a policy violation to access your own
        // content.
        isEmbedded = true;
    }

    return (
        isEmbedded &&
        toLower(referrer.protocol) === HTTPS_PROTOCOL &&
        isMobileTwitchDomain(referrer.host)
    );
}

function isMobileTwitchDomain(host) {
    const sanitizedHost = toLower(host);

    return (
        sanitizedHost === MOBILE_HOSTNAME ||
        endsWith(sanitizedHost, `.${MOBILE_HOSTNAME}`) // mweb staging: '*.m.twitch.tv'
    );
}

