import {
    ACTION_SET_EXTENSIONS,
    ACTION_EXTENSION_ACTIVATED,
    ACTION_FETCHING_EXTENSIONS,
    ACTION_CLEAR_EXTENSIONS,
    ACTION_REQUEST_TOGGLE_EXTENSION_IDENTITY,
    ACTION_SET_EXTENSION_IDENTITY,
    ACTION_SET_EXTENSION_MODAL,
    ACTION_PURCHASE_INTENT_RECEIVED,
    EXTENSIONS_NOT_LOADED,
    EXTENSIONS_LOADED,
    EXTENSIONS_LOADING,
    EXTENSIONS_RELOADING,
} from 'actions/extensions';
import {
    EXTENSION_PERMISSION_STATE_GRANTED,
    EXTENSION_PERMISSION_STATE_NONE,
} from 'util/extensions';
import assign from 'lodash/assign';
import merge from 'lodash/merge';
import PropTypes from 'prop-types';

export const DEFAULT_EXTENSIONS = {
    loadingState: EXTENSIONS_NOT_LOADED,
    channel: undefined,
    activeOverlayExtensions: [],
    modalRequest: undefined,
    purchaseIntent: null,
};

export const EXTENSION_PROP_TYPE = PropTypes.shape({
    id: PropTypes.string.isRequired,
    name: PropTypes.string.isRequired,
    summary: PropTypes.string.isRequired,
    anchor: PropTypes.string.isRequired,
    version: PropTypes.string.isRequired,
    viewerUrl: PropTypes.string.isRequired,
    lastUserIdentityLinkState: PropTypes.bool.isRequired,
    supportsIdentityLinking: PropTypes.bool.isRequired,
    token: PropTypes.shape({
        token: PropTypes.string.isRequired,
        permissionsState: PropTypes.string.isRequired,
        role: PropTypes.string.isRequired,
    }),
}).isRequired;

export const EXTENSION_MODAL_REQUEST_TYPE = PropTypes.shape({
    message: PropTypes.shape({
        action: PropTypes.string.isRequired,
        payload: PropTypes.object.isRequired,
    }),
    resultCallback: PropTypes.func.isRequired,
});

export function extensions(state = DEFAULT_EXTENSIONS, action) {
    switch (action.type) {
    case ACTION_CLEAR_EXTENSIONS:
        return assign({}, state, DEFAULT_EXTENSIONS);
    case ACTION_FETCHING_EXTENSIONS:
        return assign({}, state, {
            loadingState: EXTENSIONS_LOADING,
            channel: action.channel,
            activeOverlayExtensions: [],
        });
    case ACTION_SET_EXTENSIONS:
        return assign({}, state, {
            loadingState: EXTENSIONS_LOADED,
            channel: action.channel,
            activeOverlayExtensions: action.extensions,
        });
    case ACTION_EXTENSION_ACTIVATED:
        return assign({}, state, {
            loadingState: EXTENSIONS_RELOADING,
            channel: action.channel,
        });
    case ACTION_REQUEST_TOGGLE_EXTENSION_IDENTITY:
        return assign({}, state, {
            activeOverlayExtensions: state.activeOverlayExtensions.map(ext => {
                if (ext.id !== action.extensionId) {
                    return ext;
                }

                const currentIsIdentityLinked = ext.token.permissionsState === EXTENSION_PERMISSION_STATE_GRANTED;
                return assign({}, ext, {
                    lastUserIdentityLinkState: !currentIsIdentityLinked,
                });
            }),
        });
    case ACTION_SET_EXTENSION_IDENTITY:
        return assign({}, state, {
            activeOverlayExtensions: state.activeOverlayExtensions.map(ext => {
                if (ext.id !== action.extensionId) {
                    return ext;
                }

                return merge({}, ext, {
                    lastUserIdentityLinkState: action.isLinked,
                    token: {
                        permissionsState: action.isLinked ?
                            EXTENSION_PERMISSION_STATE_GRANTED :
                            EXTENSION_PERMISSION_STATE_NONE,
                    },
                });
            }),
        });
    case ACTION_SET_EXTENSION_MODAL:
        return assign({}, state, {
            modalRequest: action.modalRequest,
        });
    case ACTION_PURCHASE_INTENT_RECEIVED:
        return assign({}, state, {
            purchaseIntent: action.purchaseIntent,
        });
    default:
        return state;
    }
}
