import assign from 'lodash/assign';
import pick from 'lodash/pick';
import * as PlaybackActions from '../actions/playback';
import { ACTION_SET_STREAM } from '../actions/stream';
import { ACTION_SET_COLLECTION, ACTION_CLEAR_COLLECTION, ACTION_LOADED_LAST_COLLECTION_ITEM,
         ACTION_LOADED_COLLECTION_ITEM } from '../actions/collection';
import { nullContentStream } from '../stream/null';

export const TRANSITION_TYPE_COLLECTION = 'transition collection';
export const TRANSITION_TYPE_RECOMMENDATIONS = 'transition recommendations';

const DEFAULT_PLAYBACK = {
    autoplay: true,
    buffer: {
        start: 0,
        end: 0,
    },
    bufferEmpties: 0,
    currentTime: 0,
    startTime: 0,
    startTimeSet: false,
    duration: 0,
    ended: false,
    hasPlayed: false,
    isLoading: false,
    isSeeking: false,
    muted: false,
    paused: false,
    playing: false,
    playbackRate: 1.0,
    restrictedQualityError: false,
    volume: 0.5,
    transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS,
    contentShowing: false,
};

const PLAYING_STATE = {
    playing: true,
    hasPlayed: true,
    paused: false,
    ended: false,
    isSeeking: false,
};

const WAITING_STATE = {
    ended: false,
};

const PAUSE_STATE = {
    playing: false,
    paused: true,
    ended: false,
};

const ENDED_STATE = {
    playing: false,
    paused: true,
    ended: true,
};

const CONTENT_SHOWING_STATE = {
    contentShowing: true,
};

const NEW_STREAM_STATE = {
    playing: false,
    hasPlayed: false,
    contentShowing: false,
    bufferEmpties: 0,
    startTime: 0,
    startTimeSet: false,
    isSeeking: false,
    restrictedQualityError: false,
};

const PLAYBACK_KEYS = Object.keys(DEFAULT_PLAYBACK);

/**
 * Store reducer for Playback
 *
 * @param {Object} parameters
 * @param {Action} action
 * @return {Object}
 */
// eslint-disable-next-line complexity
export function playback(playback = DEFAULT_PLAYBACK, action) {
    switch (action.type) {
    case PlaybackActions.ACTION_INITIALIZE_PLAYBACK_SETTINGS:
        return assign({}, playback, pick(action.playback, PLAYBACK_KEYS));
    case ACTION_SET_STREAM:
        if (action.stream === nullContentStream) {
            return assign({}, playback, NEW_STREAM_STATE, {
                paused: false,
            });
        }
        return assign({}, playback, NEW_STREAM_STATE);
    case PlaybackActions.ACTION_PLAYING:
        return assign({}, playback, PLAYING_STATE, CONTENT_SHOWING_STATE);
    case PlaybackActions.ACTION_WAITING:
        return assign({}, playback, WAITING_STATE);
    case PlaybackActions.ACTION_PAUSE:
        return assign({}, playback, PAUSE_STATE);
    case PlaybackActions.ACTION_ENDED:
        return assign({}, playback, ENDED_STATE);
    case PlaybackActions.ACTION_PLAYER_MUTED:
        return assign({}, playback, {
            muted: action.muted,
        });
    case PlaybackActions.ACTION_VOLUME_CHANGED:
        return assign({}, playback, {
            volume: action.volume,
        });
    case PlaybackActions.ACTION_UPDATE_PLAYBACK_DURATION:
        return assign({}, playback, pick(action.playback, PLAYBACK_KEYS));
    case PlaybackActions.ACTION_QUALITY_RESTRICTED_ERROR:
        return assign({}, playback, { restrictedQualityError: true });
    case PlaybackActions.ACTION_CLEAR_QUALITY_RESTRICTED_ERROR:
        return assign({}, playback, { restrictedQualityError: false });
    case PlaybackActions.ACTION_SET_AUTOPLAY_STATE:
        return assign({}, playback, {
            autoplay: action.playback.autoplay,
            paused: !action.playback.autoplay,
        });
    case PlaybackActions.ACTION_SET_LOADING:
        return assign({}, playback, { isLoading: action.isLoading });
    case PlaybackActions.ACTION_PLAYER_SEEKING:
        return assign({}, playback, { isSeeking: true });
    case PlaybackActions.ACTION_PLAYER_SEEKED:
        return assign({}, playback, { isSeeking: false });
    case PlaybackActions.ACTION_UPDATE_CURRENT_TIME:
        return assign({}, playback, { currentTime: action.currentTime });
    case PlaybackActions.ACTION_UPDATE_BUFFER_VALUES:
        return assign({}, playback, {
            buffer: {
                start: action.start,
                end: action.end,
            },
        });
    case PlaybackActions.ACTION_INCREMENT_BUFFER_EMPTIES:
        return assign({}, playback, { bufferEmpties: playback.bufferEmpties + 1 });
    case PlaybackActions.ACTION_CONTENT_IS_SHOWING:
        return assign({}, playback, CONTENT_SHOWING_STATE);
    case ACTION_LOADED_COLLECTION_ITEM:
        return assign({}, playback, { transitionScheme: TRANSITION_TYPE_COLLECTION });
    case ACTION_LOADED_LAST_COLLECTION_ITEM:
        return assign({}, playback, { transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS });
    case ACTION_SET_COLLECTION:
        return assign({}, playback, { transitionScheme: TRANSITION_TYPE_COLLECTION });
    case ACTION_CLEAR_COLLECTION:
        return assign({}, playback, { transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS });
    case PlaybackActions.ACTION_PLAYBACK_RATE_CHANGED:
        return assign({}, playback, { playbackRate: action.playbackRate });
    case PlaybackActions.ACTION_SET_START_TIME:
        return assign({}, playback, {
            startTime: action.startTime,
            startTimeSet: true,
        });
    default:
        return playback;
    }
}

export function isPlayingSound({ playback }) {
    return !playback.muted && playback.volume !== 0;
}
