import assign from 'lodash/assign';
import omit from 'lodash/omit';
import merge from 'lodash/merge';
import pick from 'lodash/pick';
import { ACTION_VOD_INIT_RESUME, ACTION_VOD_CANCEL_RESUME, ACTION_SET_RESUME_TIMES,
         ACTION_VOD_SET_RESUME_TIME, ACTION_LIVESTREAM_CANCEL_RESUME,
         ACTION_LIVESTREAM_SET_RESUME_TIME, ACTION_VOD_SET_USER,
         ACTION_VOD_POST_BACKEND_TIME, ACTION_VOD_SET_IS_SEEKED } from '../actions/resume-watch';
import { ACTION_SET_STREAM } from '../actions/stream';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from '../stream/twitch-vod';
import { LiveTwitchContentStream, CONTENT_MODE_LIVE } from '../stream/twitch-live';
import { ACTION_USER_LOGGED_IN } from '../actions/user';

export const UPDATE_INTERVAL_VOD = 20; // s
export const UPDATE_INTERVAL_LIVESTREAM = 600; // s

export const INIT_UPDATE_OFFSET_VOD = 20000; // ms
export const MAX_INIT_UPDATE_OFFSET_LIVESTREAM = 480000; // ms

const DEFAULT_RESUME_WATCH = {
    times: {},
    watch: {},
    streamTimes: {},
    lastTimeStamp: 0,
    userId: null,
    initUpdateOffset: 0,
    updateInterval: 0,
    isSeeked: false,
};

/**
 * Store reducer for VOD resume watching times.
 *
 * @param {Object} resumeWatching
 * @param {Action} action
 * @return {Object}
 */
// eslint-disable-next-line complexity
export function resumeWatch(resumeWatching = DEFAULT_RESUME_WATCH, action) {
    switch (action.type) {
    case ACTION_VOD_INIT_RESUME:
        return pick(action, Object.keys(DEFAULT_RESUME_WATCH));
    case ACTION_VOD_SET_USER:
        return assign({}, resumeWatching, { userId: action.userId });
    case ACTION_USER_LOGGED_IN:
        return assign({}, resumeWatching, { userId: action.userInfo.id });
    case ACTION_SET_RESUME_TIMES:
        return assign({}, resumeWatching, action.resumeTimes.reduce((map, resumeTime) => {
            /* eslint-disable no-param-reassign */
            const { video_id: videoID, position, type } = resumeTime;
            if (type === CONTENT_MODE_VOD) {
                map.times[videoID] = position;
                map.watch[videoID] = true;
            }
            if (type === CONTENT_MODE_LIVE) {
                map.streamTimes[videoID] = position;
            }
            return map;
            /* eslint-enable no-param-reassign */
        }, {
            times: {},
            watch: {},
            streamTimes: {},
        }));
    case ACTION_VOD_SET_RESUME_TIME:
        return merge({}, resumeWatching, {
            times: {
                [action.videoID]: action.time,
            },
            watch: {
                [action.videoID]: true,
            },
        });
    case ACTION_VOD_CANCEL_RESUME:
        return assign({}, resumeWatching, {
            times: omit(resumeWatching.times, action.videoID),
            watch: omit(resumeWatching.watch, action.videoID),
        });
    case ACTION_LIVESTREAM_SET_RESUME_TIME:
        return merge({}, resumeWatching, {
            streamTimes: {
                [action.broadcastID]: action.time,
            },
        });
    case ACTION_LIVESTREAM_CANCEL_RESUME:
        return assign({}, resumeWatching, {
            streamTimes: omit(resumeWatching.streamTimes, action.broadcastID),
        });
    case ACTION_VOD_POST_BACKEND_TIME:
        return assign({}, resumeWatching, { lastTimeStamp: action.lastTimeStamp });
    case ACTION_VOD_SET_IS_SEEKED:
        return assign({}, resumeWatching, { isSeeked: action.isSeeked });
    case ACTION_SET_STREAM: {
        return assign({}, resumeWatching, getUpdateFrequency(action.stream), {
            isSeeked: false,
            lastTimeStamp: 0,
        });
    }
    default:
        return resumeWatching;
    }
}

function getUpdateFrequency(stream) {
    const frequency = {
        updateInterval: DEFAULT_RESUME_WATCH.updateInterval,
        initUpdateOffset: DEFAULT_RESUME_WATCH.initUpdateOffset,
    };

    if (stream instanceof VODTwitchContentStream) {
        frequency.updateInterval = UPDATE_INTERVAL_VOD;
        frequency.initUpdateOffset = INIT_UPDATE_OFFSET_VOD;
    } else if (stream instanceof LiveTwitchContentStream) {
        frequency.updateInterval = UPDATE_INTERVAL_LIVESTREAM;
        frequency.initUpdateOffset = Math.ceil(Math.random() * MAX_INIT_UPDATE_OFFSET_LIVESTREAM);
    }

    return frequency;
}
