import { stringify } from 'query-string';
import { AbstractStream } from './stream';
import { usherHost as USHER_HOST } from '../settings';
import * as DeviceId from 'util/device-id';

export const CONTENT_MODE_PROVIDED = 'provided';

/**
 * Content stream class representing a stream with a provided URL.
 */
export class ProvidedContentStream extends AbstractStream {
    /**
     * @param {URI} contentUrl
     *        Usher URL associated with the content
     */
    constructor({ customerId, contentId, usherHostOverride }) {
        super(contentId);

        this._customerId = customerId;
        this._contentId = contentId;
        this._usherHostOverride = usherHostOverride;
    }

    /**
     * @return {String}
     */
    get contentType() {
        return CONTENT_MODE_PROVIDED;
    }

    /**
     * Accessor for the channel property, which is undefined in ProvidedContentStream
     * @return {String}
     */
    get channel() {
        return;
    }

    /**
     * Accessor for the videoId property, which is undefined in ProvidedContentStream
     * @return {String}
     */
    get videoId() {
        return;
    }

    /**
     * Accessor for the contentId property
     * @return {String}
     */
    get contentId() {
        return this._contentId;
    }

    /**
     * Accessor for the customerId property
     * @return {String}
     */
    get customerId() {
        return this._customerId;
    }

    /**
     * Accessor for the contentUrl property
     * @return {String}
     */
    get _baseStreamUrl() {
        const host = this._usherHostOverride && this._usherHostOverride !== '' ? this._usherHostOverride : USHER_HOST;
        return `${host}/api/lvs/hls/lvs.${this._customerId}.${this._contentId}.m3u8`;
    }

    /**
     * Stubbed accessor that immediately resolves the accessToken, which is not needed for ProvidedContentStream
     */
    get accessToken() {
        return Promise.resolve();
    }

    /**
     * Stubbed as it is not needed for ProvidedContentStream
     */
    resetNAuthToken() {
    }

    /**
     * Builds object describing additional parameters to append to stream URL
     *
     * @return {Object}
     */
    _buildUsherParams() {
        return {
            /* eslint-disable camelcase */
            allow_source: true,
            device_id: DeviceId.get(false),
            /* eslint-enable camelcase */
        };
    }

    /**
     * Gets the master manifest URL associated with this livestream.
     *
     * @return {Promise}
     */
    get streamUrl() {
        const params = this._buildUsherParams();
        return Promise.resolve(`${this._baseStreamUrl}?${stringify(params)}`);
    }

    /**
     * Stub method to fetch nauthToken as it is not needed for ProvidedContentStream
     */
    _fetchNAuthToken() {
        return Promise.reject('Error: Attempted NAuth fetch for ProvidedContentStream');
    }

    /**
     * Stub method to override access token fetching logic. Not needed for ProvidedContentStream
     */
    _fetchAccessToken() {
        return Promise.reject('Error: Attempted Access Token fetch for ProvidedContentStream');
    }

    /**
     * Get the master manifest URL associated with this content to be used with Chromecast
     * Chromecast does not support 1080p, so remove `allow_source`
     *
     * @return {String}
     */
    get castStreamUrl() {
        const params = this._buildUsherParams();
        // eslint-disable-next-line camelcase
        params.allow_source = undefined;
        return Promise.resolve(`${this._baseStreamUrl}?${stringify(params)}`);
    }
}
