import { stringify } from 'query-string';
import bowser from 'bowser';

/**
 * Content stream base class for common features and functionality between content types
 */
export class AbstractStream {
    /**
     * @param {String} contentId
     *        channelName or vodId
     * @param {Promise<String>} oAuthToken
     *        API response for OAuth token associated with the current viewer
     * @param {Object} usherParams
     *        Additional parameters specific to Usher that should be appended
     *        to the stream URL.
     * @param {Object} accessTokenParams
     *        Additional parameters to send to the access token request endpoint
     * @param {Object} experimentSettings
     *        Experiment assignments
     */
    // eslint-disable-next-line no-unused-vars
    constructor(contentId, oAuthToken, usherParams, accessTokenParams, experimentSettings) {
        this._contentId = contentId;
        this._oAuthToken = oAuthToken;
        this._usherParams = usherParams;
        this._accessTokenParams = accessTokenParams;
        this._experimentSettings = experimentSettings;
        this._restrictedBitrates = [];
    }

    /**
     * Presents an error in console if an invalid property or method is called such as `VODTwitchContentStream.channel`,
     * or directly instantiating `AbstractStream` and calling methods on that instance.
     *
     * @param {String} operation - the invalid method/property's name
     */
    _handleInvalidOp(operation) {
        let reason;
        if (this.constructor.name === 'AbstractStream') {
            reason = 'AbstractStream should not be instantiated directly';
        } else {
            reason = `${this.constructor.name}.${operation} is not implemented`;
        }
        // eslint-disable-next-line no-console
        console.error(`Invalid operation: ${reason}`);
    }

    /**
     * Placeholder: type of content ('live' or 'vod')
     *
     * @return {String}
     */
    get contentType() {
        this._handleInvalidOp('contentType');
        return '';
    }

    /**
     * Placeholder: channel name associated with stream if content type is live
     *
     * @return {String}
     */
    get channel() {
        this._handleInvalidOp('channel');
        return '';
    }

    /**
     * Placeholder: vod id associated with stream if content type is VOD
     *
     * @return {String}
     */
    get videoId() {
        this._handleInvalidOp('videoId');
        return '';
    }

    /**
     * Placeholder: customer id associated with stream if content type is provided
     *
     * @return {String}
     */
    get customerId() {
        this._handleInvalidOp('customerId');
        return '';
    }

    /**
     * Placeholder: content id associated with stream if content type is provided
     *
     * @return {String}
     */
    get contentId() {
        this._handleInvalidOp('contentId');
        return '';
    }

    /**
     * Placeholder: content url associated with stream if content type is provided
     *
     * @return {String}
     */
    get contentUrl() {
        this._handleInvalidOp('contentUrl');
        return '';
    }

    /**
     * Qualities that are not allowed
     *
     * @return {Array<String>}
     */
    get restrictedBitrates() {
        return this._restrictedBitrates;
    }

    /**
     * Placeholder: function that gets the access token object
     * Check subclasses for specific implementations
     *
     * @return {Promise}
     *         Resolves with the access token (object)
     */
    _fetchAccessToken() {
        this._handleInvalidOp('_fetchAccessToken');
        return Promise.resolve({});
    }

    /**
     * Fetches access token
     *
     * @return {Promise<Object>}
     */
    get accessToken() {
        let expired = false;
        try {
            if (this._nAuth) {
                const token = JSON.parse(this._nAuth.token);
                expired = Date.now() >= token.expires * 1000; // token.expires is in seconds
            }
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error(`Unable to JSON.parse nAuth.token: ${this._nAuth.token}`);
        }

        if (!this._nAuthPromise || expired) {
            this._nAuthPromise = this._fetchNAuthToken();
        }

        return this._nAuthPromise;
    }

    /**
     * Resets the current NAuthToken, forces a new fetch for
     * the NAuth Token.
     *
     * @return {Promise<Object>}
     */
    resetNAuthToken() {
        this._nAuthPromise = this._fetchNAuthToken();
    }

    _fetchNAuthToken() {
        return this._oAuthToken.
            then(token => this._fetchAccessToken(this._contentId, token, this._accessTokenParams)).
            then(nAuth => {
                this._nAuth = nAuth;
                return nAuth;
            });
    }

    /**
     * Placeholder: gets the master manifest URL without params
     * Check subclasses for specific implementations
     *
     * @return {String}
     */
    get _baseStreamUrl() {
        this._handleInvalidOp('_baseStreamUrl');
        return '';
    }

    /**
     * Placeholder: builds object that describes parameters to append to stream URL
     * Check subclasses for specific implementations
     *
     * @return {Object}
     *         Additional parameters and what their values should be set to
     */
    _buildUsherParams(_) {
        this._handleInvalidOp('_buildUsherParams');
        return {};
    }

    _commonExperimentParams() {
        return Promise.all([
            this.accessToken,
            this._experimentSettings.realtimeQos,
            this._experimentSettings.fastBread,
        ]).then(([
            accessToken,
            realtimeQos,
            fastBread,
        ]) => {
            const params = {};
            params.rtqos = realtimeQos;
            // eslint-disable-next-line camelcase
            params.fast_bread = fastBread === 'treatment' && bowser.chrome;
            this._restrictedBitrates = JSON.parse(accessToken.token).chansub.restricted_bitrates;
            return params;
        });
    }

    /**
     * Placeholder: gets the master manifest URL associated with provided content
     * Check subclasses for specific implementations
     *
     * @return {String}
     */
    get streamUrl() {
        this._handleInvalidOp('streamUrl');
        return '';
    }

    /**
     * Gets the master manifest URL associated with this video to be used with Chromecast
     * Chromecast does not support 1080p so remove `allow_source`
     *
     * @return {String}
     */
    get castStreamUrl() {
        return this.accessToken.then(accessToken => {
            // If there is no token on the accessToken object, something went horribly wrong
            // or the method being called directly from `AbstractStream`, which is invalid.
            if (!accessToken.token) {
                this._handleInvalidOp('castStreamUrl');
                return '';
            }

            const params = this._buildUsherParams(accessToken);
            delete params.allow_source;
            this._restrictedBitrates = JSON.parse(accessToken.token).chansub.restricted_bitrates;
            return `${this._baseStreamUrl}?${stringify(params)}`;
        });
    }
}
