import assign from 'lodash/assign';
import extend from 'lodash/extend';
import includes from 'lodash/includes';
import { AD_STITCHING_GEOS, PREROLL_TIMEOUT } from '../ads/surestream/consts';
import { userInfo } from '../api';
import { stringify } from 'query-string';
import { usherHost as USHER_HOST } from '../settings';
import { fetchChannelAccessToken } from '../util/tokens';
import { AbstractStream } from './stream';

export const CONTENT_MODE_LIVE = 'live';

/**
 * Content stream class representing live streams.
 */
export class LiveTwitchContentStream extends AbstractStream {
    /**
     * @param {String} channelName
     *        Canonicalized name of the channel being watched
     * @param {Promise<String>} oAuthToken
     *        API response for OAuth token associated with the current viewer
     *        watching this live stream.
     * @param {Object} usherParams
     *        Additional parameters specific to Usher that should be appended
     *        to the stream URL.
     * @param {Object} accessTokenParams
     *        Additional parameters to send to the access token request endpoint
     * @param {Object} experimentSettings
     *        Experiment assignments
     */
    // eslint-disable-next-line no-unused-vars
    constructor(channelName, oAuthToken, usherParams, accessTokenParams, experimentSettings) {
        super(...arguments);
        this._channelName = channelName;
    }

    /**
     * @return {String}
     */
    get contentType() {
        return CONTENT_MODE_LIVE;
    }

    /**
     * @return {String}
     */
    get channel() {
        return this._channelName;
    }

    /**
     * Makes function that gets the access token object for live stream
     *
     * @param {String} channelName
     * @param {Promise<String>} oAuthToken
     *        API response for OAuth token associated with the current viewer
     * @param {Object} accessTokenParams
     *        Additional parameters to send to the access token request endpoint
     * @return {Promise}
     *         Resolves with the access token (object)
     */
    _fetchAccessToken(channelName, oAuthToken, accessTokenParams) {
        return fetchChannelAccessToken(channelName, oAuthToken, accessTokenParams);
    }

    /**
     * Gets the master manifest URL for the live stream without params
     *
     * @return {String}
     */
    get _baseStreamUrl() {
        return `${USHER_HOST}/api/channel/hls/${this._channelName.toLowerCase()}.m3u8`;
    }

    /**
     * Builds object describing additional parameters to append to stream URL
     *
     * @return {Object}
     */
    _buildUsherParams(accessToken) {
        return extend({
            token: accessToken.token,
            sig: accessToken.sig,
            // eslint-disable-next-line camelcase
            allow_source: true,
        }, this._usherParams);
    }

    /**
     * Gets the master manifest URL associated with this livestream.
     *
     * @return {Promise}
     */
    get streamUrl() {
        return Promise.all([
            this.accessToken,
            this._commonExperimentParams(),
            userInfo(),
        ]).then(([accessToken, experimentParams, userInfo]) => {
            const params = this._buildUsherParams(accessToken);
            const surestreamParams = {
                /* eslint-disable camelcase */
                baking_bread: false,
                baking_brownies: false,
                baking_brownies_timeout: PREROLL_TIMEOUT,
                /* eslint-enable camelcase */
            };

            if (includes(AD_STITCHING_GEOS, userInfo.geo)) {
                /* eslint-disable camelcase */
                surestreamParams.baking_bread = true;
                surestreamParams.baking_brownies = true;
                /* eslint-enable camelcase */
            }

            assign(params, experimentParams, surestreamParams);

            // eslint-disable-next-line max-len
            return `${USHER_HOST}/api/channel/hls/${this._channelName.toLowerCase()}.m3u8?${stringify(params)}`;
        });
    }
}
