import assign from 'lodash/assign';
import extend from 'lodash/extend';
import includes from 'lodash/includes';
import { AD_STITCHING_GEOS, PREROLL_TIMEOUT } from '../ads/surestream/consts';
import { userInfo } from '../api';
import { stringify } from 'query-string';
import { usherHost as USHER_HOST } from '../settings';
import { fetchVideoAccessToken } from '../util/tokens';
import { AbstractStream } from './stream';

export const CONTENT_MODE_VOD = 'vod';

/**
 * Content stream class representing prerecorded video content.
 */
export class VODTwitchContentStream extends AbstractStream {
    /**
     * @param {String} vodId
     *        ID associated with the vod (e.g. "v1234567890")
     * @param {Promise<String>} oAuthToken
     *        API response for OAuth token associated with the current viewer
     *        watching this vod stream.
     * @param {Object} usherParams
     *        Additional parameters specific to Usher that should be appended
     *        to the stream URL.
     * @param {Object} accessTokenParams
     *        Additional parameters to send to the access token request endpoint
     * @param {Object} experimentSettings
     *        Experiment assignments
     */
    // eslint-disable-next-line no-unused-vars
    constructor(vodId, oAuthToken, usherParams, accessTokenParams, experimentSettings) {
        super(...arguments);
        this._vodId = vodId;
    }

    /**
     * @return {String}
     */
    get contentType() {
        return CONTENT_MODE_VOD;
    }

    /**
     * @return {String}
     */
    get videoId() {
        return this._vodId;
    }

    /**
     * Makes function that gets the access token object for VOD stream
     *
     * @param {String} channelName
     * @param {Promise<String>} oAuthToken
     *        API response for OAuth token associated with the current viewer
     * @param {Object} accessTokenParams
     *        Additional parameters to send to the access token request endpoint
     * @return {Promise}
     *         Resolves with the access token (object)
     */
    _fetchAccessToken(vodId, oAuthToken, accessTokenParams) {
        return fetchVideoAccessToken(vodId, oAuthToken, accessTokenParams);
    }

    /**
     * Gets the master manifest URL for the VOD stream without params
     *
     * @return {String}
     */
    get _baseStreamUrl() {
        return `${USHER_HOST}/vod/${this._vodId.substring(1)}.m3u8`;
    }

    /**
     * Builds object describing additional parameters to append to stream URL
     *
     * @param {Object} accessToken
     *
     * @return {Object}
     *         Additional parameters and what their values should be set to
     */
    _buildUsherParams(accessToken) {
        return extend({
            nauth: accessToken.token,
            nauthsig: accessToken.sig,
            // eslint-disable-next-line camelcase
            allow_source: true,
        }, this._usherParams);
    }

    /**
     * Gets the master manifest URL associated with this video.
     *
     * @return {Promise}
     */
    get streamUrl() {
        return Promise.all([
            this.accessToken,
            this._commonExperimentParams(),
            userInfo(),
        ]).then(([accessToken, experimentParams, userInfo]) => {
            const params = this._buildUsherParams(accessToken);
            const surestreamParams = {
                /* eslint-disable camelcase */
                baking_bread: false,
                baking_brownies: false,
                baking_brownies_timeout: PREROLL_TIMEOUT,
                /* eslint-enable camelcase */
            };

            if (includes(AD_STITCHING_GEOS, userInfo.geo)) {
                /* eslint-disable camelcase */
                surestreamParams.baking_bread = true;
                surestreamParams.baking_brownies = true;
                /* eslint-enable camelcase */
            }

            assign(params, experimentParams, surestreamParams);

            return `${this._baseStreamUrl}?${stringify(params)}`;
        });
    }
}
