import { videoInfo } from './../api';
import groupBy from 'lodash/groupBy';
import assign from 'lodash/assign';
import map from 'lodash/map';
import flatten from 'lodash/flatten';

export const TYPE_GAME_METADATA = 'game metadata';

// VCA Markers Base Url
const VCA_MARKER_JSON_BASE_URL = 'https://clipmn.twitch.tv/prod/';

// VCA Marker Games
const GAME_LEAGUE_OF_LEGENDS = 'lol';
const GAME_HEARTHSTONE = 'hs';
const GAME_COUNTER_STRIKE = 'csgo';
const GAME_ROCKET_LEAGUE = 'rl';

// VCA Tag Types
const TAG_TYPE_LEAGUE_OF_LEGENDS_MATCH = 'leagueOfLegendTags';
const TAG_TYPE_HEARTHSTONE_MATCH = 'hearthStoneMatchTags';
const TAG_TYPE_COUNTER_STRIKE_MATCH = 'counterStrikeMatchTags';
const TAG_TYPE_ROCKET_LEAGUE_MATCH = 'rocketLeagueMatchTags';
const TAG_TYPE_NONE = 'nonTypedTags';

// VCA Marker Normalizing Function Maps
const VCA_MARKER_NORMALIZING_FUNCTION_MAP = {
    [TAG_TYPE_LEAGUE_OF_LEGENDS_MATCH]: getNormalizedMatchMarker,
    [TAG_TYPE_ROCKET_LEAGUE_MATCH]: getNormalizedMatchMarker,
    [TAG_TYPE_COUNTER_STRIKE_MATCH]: getNormalizedMatchMarker,
    [TAG_TYPE_HEARTHSTONE_MATCH]: (tag, index, data) => {
        return assign(getNormalizedMatchMarker(tag, index, data), {
            info: `<strong>${tag.game_data.characters[0]}</strong> vs ${tag.game_data.characters[1]}`,
        });
    },
    [TAG_TYPE_NONE]: () => null,
};

const NULL_MARKER_THUMBNAIL = {
    imageURL: '',
    x: 0,
    y: 0,
    width: 0,
    height: 0,
    cols: 0,
};

export function getVCAMarkers(videoID) {
    const fetchAvailableVCAChannels = fetch(`${VCA_MARKER_JSON_BASE_URL}ids.json`).then(response => response.json());

    return Promise.all([fetchAvailableVCAChannels, videoInfo(videoID)]).
        then(([availableVCAChannels, info]) => {
            if (availableVCAChannels.indexOf(info.channel._id) === -1) {
                return Promise.reject(new Error('Current channel does not have VCA markers'));
            }
            return fetch(`${VCA_MARKER_JSON_BASE_URL}${videoID}.json`).then(response => response.json());
        }).
        then(normalizedVCAMarkers).
        catch((error = new Error('Could not retrieve and/or normalize VCA markers')) => {
            // eslint-disable-next-line no-console
            console.warn(error.message);
            return [];
        });
}

/**
 * Converts the marker api response from Video Content Analysis
 * into normalized markers for the player to use.
 *
 * @param {Object} raw marker data from VCA
 * @return {Array} array of normalized markers to set in statestore
 */
function normalizedVCAMarkers(rawMarkerData) {
    if (rawMarkerData === null) {
        return Promise.reject(new Error('Raw VCA marker data not available'));
    }

    const tagsGroupedByType = groupBy(rawMarkerData.data.tags, getTagType);

    const normalizedMarkers = flatten(map(tagsGroupedByType, (tags, type) => {
        return tags.
            sort((t1, t2) => t1.start_sec - t2.start_sec).
            map((tag, i) => {
                const markerNormalizer = VCA_MARKER_NORMALIZING_FUNCTION_MAP[type];
                return markerNormalizer(tag, i, rawMarkerData.data);
            });
    }));

    return normalizedMarkers.
        filter(m => m !== null).
        sort((m1, m2) => m1.startTime - m2.startTime);
}

function getTagType(tag) {
    switch (tag.game_type) {
    case GAME_LEAGUE_OF_LEGENDS:
        if (tag.game_data.type === 0) {
            return TAG_TYPE_LEAGUE_OF_LEGENDS_MATCH;
        }
        break;
    case GAME_ROCKET_LEAGUE:
        if (tag.game_data.type === 0) {
            return TAG_TYPE_ROCKET_LEAGUE_MATCH;
        }
        break;
    case GAME_COUNTER_STRIKE:
        if (tag.game_data.type === 0) {
            return TAG_TYPE_COUNTER_STRIKE_MATCH;
        }
        break;
    case GAME_HEARTHSTONE:
        if (tag.game_data.type === 0) {
            return TAG_TYPE_HEARTHSTONE_MATCH;
        }
        break;
    }

    return TAG_TYPE_NONE;
}

function getNormalizedMatchMarker(tag, index, data) {
    const marker = {
        id: tag.id,
        title: `Match ${index + 1}`,
        info: '',
        startTime: tag.start_sec,
        thumbnail: getNormalizedMarkerThumbnail(tag, data),
        type: TYPE_GAME_METADATA,
    };

    return marker;
}

function getNormalizedMarkerThumbnail(tag, data) {
    if (tag.thumbnail_index === null) {
        return NULL_MARKER_THUMBNAIL;
    }

    return {
        imageURL: `${VCA_MARKER_JSON_BASE_URL}${data.thumbnail_sheet}`,
        // x, y: pixel coordinates in the marker sprite sheet
        // tag.thumbnail_index is [y coordinate of thumbnail, x coordinate of thumbnail]
        // data.thumbnail_size is [thumbnail (px), thumbnail height(px)]
        x: tag.thumbnail_index[1] * data.thumbnail_size[0],
        y: tag.thumbnail_index[0] * data.thumbnail_size[1],
        width: data.thumbnail_size[0],
        height: data.thumbnail_size[1],
        cols: data.sheet_dimensions[1],
    };
}
