import { krakenRequestv5 } from './../api';
import flatten from 'lodash/flatten';
import map from 'lodash/map';

export const TYPE_GAME_CHANGE = 'game change';

// VOD Marker Types
const VOD_MARKER_TYPE_GAME_CHANGE = 'game_changes';

// VOD Marker Normalizing Function Maps
const VOD_MARKER_NORMALIZING_FUNCTION_MAP = {
    [VOD_MARKER_TYPE_GAME_CHANGE](marker) {
        return {
            title: '',
            info: '',
            startTime: marker.time,
            thumbnail: getMarkerThumbnail(marker),
            type: TYPE_GAME_CHANGE,
        };
    },
};

const DEFAULT_NORMALIZING_FUNCTION = () => null;

export function getVODMarkers(videoID) {
    const videoIDString = videoID.substring(videoID.indexOf('v') + 1);

    return krakenRequestv5(`videos/${videoIDString}/markers`).
        then(data => normalizedVODMarkers(data));
}

function normalizedVODMarkers(rawData = { markers: [] }) {
    if (rawData === null) {
        return Promise.reject(new Error('Raw VOD marker data not available'));
    }

    if (rawData.markers === undefined || rawData.markers === null) {
        return Promise.reject(new Error('Raw VOD marker data is misformed'));
    }

    const normalizedMarkers = flatten(map(rawData.markers, (markerGroup, markerType) => {
        // DEFAULT_NORMALIZING_FUNCTION returns null
        const markerNormalizer = VOD_MARKER_NORMALIZING_FUNCTION_MAP[markerType] || DEFAULT_NORMALIZING_FUNCTION;
        return markerGroup.map(markerNormalizer);
    }));

    return normalizedMarkers.
        filter(m => m !== null).
        sort((m1, m2) => m1.startTime - m2.startTime);
}

function getMarkerThumbnail(marker) {
    return {
        imageURL: `${marker.thumbnail.sheet_url}`,
        x: marker.thumbnail.column * marker.thumbnail.width,
        y: marker.thumbnail.row * marker.thumbnail.height,
        width: marker.thumbnail.width,
        height: marker.thumbnail.height,
        cols: marker.thumbnail.sheet_cols,
    };
}
