import groupBy from 'lodash/groupBy';
import map from 'lodash/map';
import values from 'lodash/values';

export const NULL_MARKER_THUMBNAIL = {
    imageURL: '',
    x: 0,
    y: 0,
    width: 0,
    height: 0,
    cols: 0,
};

// Assuming video ID's are appended on base URLs
const MARKER_JSON_BASE_URL = 'https://clipmn.twitch.tv/prod/';

// Game Types
const GAME_LEAGUE_OF_LEGEND = 'lol';
const GAME_HEARTHSTONE = 'hs';

// Tag Groups
const TAGS_LEAGUE_OF_LEGEND = 'leagueOfLegendTags';
const TAGS_HEARTHSTONE_OPENING = 'hearthStoneOpeningTags';
const TAGS_HEARTHSTONE_MATCH = 'hearthStoneMatchTags';
const TAGS_UNGROUPED = 'ungroupedTags';

// Marker Normalizing Function Maps
const MARKER_MAP = {
    [TAGS_LEAGUE_OF_LEGEND](tag, index, data) {
        const marker = {
            title: `Match ${index + 1}`,
            info: '',
            startTime: tag.start_sec,
            thumbnail: getMarkerThumbnail(tag, index, data),
        };

        return marker;
    },
    [TAGS_HEARTHSTONE_MATCH](tag, index, data) {
        const marker = {
            title: `Match ${index + 1}`,
            // eslint-disable-next-line max-len
            info: `<strong>${tag.game_data.characters[0]}</strong> vs ${tag.game_data.characters[1]}`,
            startTime: tag.start_sec,
            thumbnail: getMarkerThumbnail(tag, index, data),
        };

        return marker;
    },
};

const DEFAULT_MARKER_MAP = () => null;

const getMarkerThumbnail = (tag, index, data) => {
    if (tag.thumbnail_index === null) {
        return NULL_MARKER_THUMBNAIL;
    }

    return {
        imageURL: `${MARKER_JSON_BASE_URL}${data.thumbnail_sheet}`,
        x: tag.thumbnail_index[1] * data.thumbnail_size[0],
        y: tag.thumbnail_index[0] * data.thumbnail_size[1],
        width: data.thumbnail_size[0],
        height: data.thumbnail_size[1],
        cols: data.sheet_dimensions[1],
    };
};

/**
 * Timeline Metadata
 * Retrieves and normalizes all info related to timeline enhancements.
 * Encompasses:
 * 1) Markers
 * 2) Thumbnail Previews (TODO : )
 * 3) Muted Segments
 * @class
 */
export class TimelineMetadata {
    constructor(tracker, options) {
        this._analytics = tracker;
        this._options = options;
    }
    getMarkers(videoInfo) {
        return fetch(`${MARKER_JSON_BASE_URL}channels.json`).
            then(response => response.json()).
            then(channels => {
                if (channels.indexOf(videoInfo.channel.name) === -1) {
                    return Promise.reject('Current channel is not marker enabled');
                }
                return fetch(`${MARKER_JSON_BASE_URL}${videoInfo._id}.json`);
            }).
            then(response => response.json()).
            then(this._normalizedMarkers).
            catch(() => []);
    }

    _normalizedMarkers(rawMarkerData) {
        if (rawMarkerData === null) {
            return [];
        }

        const groupedTags = groupBy(rawMarkerData.data.tags, tag => {
            switch (tag.game_type) {
            case GAME_LEAGUE_OF_LEGEND:
                return TAGS_LEAGUE_OF_LEGEND;
            case GAME_HEARTHSTONE:
                switch (tag.game_data.type) {
                case 0:
                    return TAGS_HEARTHSTONE_MATCH;
                case 1:
                    return TAGS_HEARTHSTONE_OPENING;
                }
                break;
            }
            return TAGS_UNGROUPED;
        });

        return Array.prototype.concat.apply(
            [],
            values(
                map(groupedTags, (group, groupName) => {
                    return group.map((tag, i) => {
                        const markerMapper = MARKER_MAP[groupName] || DEFAULT_MARKER_MAP;
                        return markerMapper(tag, i, rawMarkerData.data);
                    });
                })
            )
        ).filter(m => m !== null).sort();
    }
}
