import React from 'react';
import PropTypes from 'prop-types';
import * as StyleMap from 'captions/captions-style-map';
import extend from 'lodash/extend';
import omit from 'lodash/omit';
import pick from 'lodash/pick';

const propTypes = {
    captionsText: PropTypes.array.isRequired,
    captionsStyle: PropTypes.shape({
        fontSize: PropTypes.number,
        font: PropTypes.string,
        fontOpacity: PropTypes.string,
        alignment: PropTypes.string,
        edge: PropTypes.string,
        verticalPosition: PropTypes.string,
        backgroundColorName: PropTypes.string,
        backgroundOpacity: PropTypes.string,
        windowColorName: PropTypes.string,
        windowOpacity: PropTypes.string,
        fontUnderline: PropTypes.string,
        fontBold: PropTypes.string,
        fontItalic: PropTypes.string,
    }).isRequired,
};

const CAPTION_ROW_CLASS = 'player-captions-row';
const CONTAINER_PROPERTIES = Object.freeze([
    'top',
    'bottom',
    'textAlign',
    'fontSize',
]);

export function ClosedCaptions({ captionsText, captionsStyle }) {
    const captionRows = captionsText.map((text, index) => {
        return (
            <div
                className={CAPTION_ROW_CLASS}
                key={`${index}-${text}`}
            >
                {text}
            </div>
        );
    });

    const {
        containerStyles,
        textStyles,
        windowStyles,
    } = parseStyles(captionsStyle);

    return (
        <div
            className="player-captions-container"
            style={containerStyles}
        >
            <div
                style={windowStyles}
                className="player-captions-window"
            >
                <div
                    className="player-captions"
                    style={textStyles}
                >
                    {captionRows}
                </div>
            </div>
        </div>
    );
}

function parseStyles(styles) {
    const newColorProperties = getFontColor(styles);
    const backgroundColorWithOpacity = getBackgroundColor(styles);
    const windowStyles = getWindowStyle(styles);

    const mergedStyle = extend(
        {},
        styles,
        newColorProperties,
        StyleMap.fontMap[styles.font],
        StyleMap.edgeMap[styles.edge],
        StyleMap.verticalPositionMap[styles.verticalPosition],
        StyleMap.fontUnderlineMap[styles.fontUnderline],
        StyleMap.fontItalicMap[styles.fontItalic],
        StyleMap.fontBoldMap[styles.fontBold],
        StyleMap.alignmentMap[styles.alignment],
        { backgroundColor: backgroundColorWithOpacity }
    );

    return {
        textStyles: omit(mergedStyle, CONTAINER_PROPERTIES),
        containerStyles: pick(mergedStyle, CONTAINER_PROPERTIES),
        windowStyles,
    };
}

function getFontColor({ fontColorName = 'white', fontOpacity = 'solid' }) {
    const colorWithOpacity = StyleMap.calculateRGBA(
        StyleMap.fontColorMap[fontColorName].fontColorValue,
        StyleMap.opacityMap[fontOpacity].opacityValue
    );

    return {
        color: colorWithOpacity,
        animation: StyleMap.opacityMap[fontOpacity].animation,
    };
}

function getBackgroundColor({ backgroundOpacity = 'solid', backgroundColorName = 'black' }) {
    return StyleMap.calculateRGBA(
        StyleMap.backgroundColorMap[backgroundColorName].backgroundColorValue,
        StyleMap.opacityMap[backgroundOpacity].opacityValue
    );
}

function getWindowStyle({ windowColorName = 'transparent', windowOpacity = 'solid' }) {
    const backgroundColor = StyleMap.calculateRGBA(
        StyleMap.backgroundColorMap[windowColorName].backgroundColorValue,
        StyleMap.opacityMap[windowOpacity].opacityValue
    );

    return { backgroundColor };
}

ClosedCaptions.propTypes = propTypes;
