import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { presetMap } from '../../../captions/captions-style-map';

const propTypes = {
    onSelect: PropTypes.func.isRequired,
    selectedValue: PropTypes.string.isRequired,
};

export const CC_PRESET_VALUES = Object.freeze(
    Object.keys(presetMap)
);

/**
 * Checks to see if a preset value is `transparent`
 *
 * @param {String} presetValue
 * @return {Boolean}
 */
export function isTransparentBG(presetValue) {
    if (typeof presetValue !== 'string') {
        return false;
    }
    return presetValue.indexOf('trans') > -1;
}

function createOnEnterKeyFn(onClick) {
    return function onKeyUp({ key }) {
        if (key === 'Enter') {
            onClick();
        }
    };
}

/* eslint-disable jsx-a11y/no-static-element-interactions */
export function PresetsPanel({ onSelect, selectedValue }) {
    const presets = CC_PRESET_VALUES.map(presetValue => {
        function onClick() {
            onSelect(presetValue);
        }
        const onKeyUp = createOnEnterKeyFn(onClick);

        const PRESET_CLASSES = classNames({
            'cc-preset-square': true,
            'cc-transparent-bg': isTransparentBG(presetValue),
            'js-cc-preset-selected': presetValue === selectedValue,
        });

        return (
            <div
                className={PRESET_CLASSES}
                onClick={onClick}
                onKeyUp={onKeyUp}
                tabIndex={-1}
                role="button"
                data-preset={presetValue}
                key={`cc-preset-${presetValue}`}
            >
                Aa
            </div>
        );
    });

    return (
        <div className="cc-panel-menu-frame cc-panel-menu-frame__presets">
            {presets}
        </div>
    );
}
/* eslint-enable */

PresetsPanel.propTypes = propTypes;
