import React from 'react';
import PropTypes from 'prop-types';
import { translate } from 'react-i18next';
import classNames from 'classnames';

const BUTTON_TEXT = 'Aa';

export const NONE = 'none';
export const RAISED = 'raised';
export const DROP = 'drop';
export const DEPRESSED = 'depressed';
export const UNIFORM = 'uniform';

const VALID_TEXT_EFFECTS = Object.freeze({
    [NONE]: true,
    [RAISED]: true,
    [DROP]: true,
    [DEPRESSED]: true,
    [UNIFORM]: true,
});

const propTypes = {
    currentTextEffect: PropTypes.string.isRequired,
    selectTextEffect: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
};

function isSelected(textEffect, currentTextEffect) {
    if (
        !VALID_TEXT_EFFECTS[textEffect] ||
        !VALID_TEXT_EFFECTS[currentTextEffect]
    ) {
        return 'cc-edge-palette__square';
    }

    return classNames({
        'pl-cc-button': true,
        'cc-edge-palette__square': true,
        [`cc-edge-palette__square--${textEffect}`]: true,
        'pl-cc-button--selected': textEffect === currentTextEffect,
    });
}

const createOnKeyUpFn = callback => ({ key }) => {
    if (key === 'Enter') {
        callback();
    }
};

/* eslint-disable jsx-a11y/no-static-element-interactions */
export function TextEffectsPaletteComponent({
    currentTextEffect,
    selectTextEffect,
    t,
}) {
    const buttons = Object.keys(VALID_TEXT_EFFECTS).map(textEffect => {
        function onClick() {
            selectTextEffect(textEffect);
        }

        const className = isSelected(textEffect, currentTextEffect);
        const onKeyUp = createOnKeyUpFn(onClick);
        const strikeThrough = textEffect === NONE ?
            <hr className="cc-edge-disabled" /> :
            null;

        return (
            <div
                className="cc-edge-palette__container"
                key={`cc-text-effect-palette.${textEffect}`}
            >
                {strikeThrough}
                <div
                    role="button"
                    tabIndex={-1}
                    htmlFor={`edge-effect-${textEffect}`}
                    className={className}
                    onClick={onClick}
                    onKeyUp={onKeyUp}
                >
                    {BUTTON_TEXT}
                </div>
            </div>
        );
    });

    return (
        <div className="cc-customize-field">
            <div>
                {t('Effect')}
            </div>
            <div className="cc-edge-palette">
                {buttons}
            </div>
        </div>
    );
}
/* eslint-enable */

TextEffectsPaletteComponent.propTypes = propTypes;
export const TextEffectsPalette = translate()(TextEffectsPaletteComponent);
