import React from 'react';
import PropTypes from 'prop-types';
import { translate } from 'react-i18next';
import classNames from 'classnames';
import map from 'lodash/map';

const UNDERLINE = 'underline';
const ITALIC = 'italic';
const BOLD = 'bold';

const VALID_TEXT_STYLES = Object.freeze({
    [BOLD]: {
        text: 'B',
        enabledPropName: 'boldText',
        styleMapName: 'fontBold',
    },
    [ITALIC]: {
        text: 'I',
        enabledPropName: 'italicText',
        styleMapName: 'fontItalic',
    },
    [UNDERLINE]: {
        text: 'U',
        enabledPropName: 'underlineText',
        styleMapName: 'fontUnderline',
    },
});

const propTypes = {
    boldText: PropTypes.bool.isRequired,
    italicText: PropTypes.bool.isRequired,
    underlineText: PropTypes.bool.isRequired,
    selectTextStyle: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
};

function isSelected(textStyle, hasBeenSelected) {
    if (
        !VALID_TEXT_STYLES[textStyle] ||
        hasBeenSelected === undefined
    ) {
        return 'cc-style-palette__square';
    }

    return classNames({
        'pl-cc-button': true,
        'cc-style-palette__square': true,
        [`cc-style-palette__square--${textStyle}`]: true,
        'pl-cc-button--selected': hasBeenSelected,
    });
}

const createOnKeyUpFn = callback => ({ key }) => {
    if (key === 'Enter') {
        callback();
    }
};

/* eslint-disable jsx-a11y/no-static-element-interactions */
export function TextStylePaletteComponent(props) {
    const { t, selectTextStyle } = props;

    const buttons = map(VALID_TEXT_STYLES, ({ text, enabledPropName, styleMapName }, textStyle) => {
        function onClick() {
            const value = props[enabledPropName] ? 'none' : textStyle;
            selectTextStyle(styleMapName, value);
        }
        const onKeyUp = createOnKeyUpFn(onClick);
        const className = isSelected(textStyle, props[enabledPropName]);

        return (
            <div
                className="cc-style-palette__container"
                key={`text-style-palette.${textStyle}`}
            >
                <div
                    role="button"
                    tabIndex={-1}
                    htmlFor={`style-${textStyle}`}
                    className={className}
                    onClick={onClick}
                    onKeyUp={onKeyUp}
                >
                    {text}
                </div>
            </div>
        );
    });

    return (
        <div className="cc-customize-field">
            <div>
                {t('Style')}
            </div>
            <div className="cc-style-palette">
                {buttons}
            </div>
        </div>
    );
}
/* eslint-enable */

TextStylePaletteComponent.propTypes = propTypes;
export const TextStylePalette = translate()(TextStylePaletteComponent);
