import PropTypes from 'prop-types';
import React from 'react';
import { CollectionSidebarItem } from './collection-sidebar-item';
import { CollectionSidebarHeader } from './collection-sidebar-header';
import { makeAnimatedComponent } from 'ui/components/common/make-animated-component';
import classNames from 'classnames';

const propTypes = {
    currentItemIndex: PropTypes.number,
    collection: PropTypes.object.isRequired,
    onSelectItem: PropTypes.func,
    onClose: PropTypes.func.isRequired,
    className: PropTypes.string,
    onMouseEnter: PropTypes.func,
    onMouseLeave: PropTypes.func,
};

const defaultProps = {
    className: '',
    currentItemIndex: -1,
    onSelectItem: () => {},
    onMouseEnter: () => {},
    onMouseLeave: () => {},
};

export const CollectionSidebarComponent = ({
    currentItemIndex,
    collection,
    onSelectItem,
    onClose,
    className,
    onMouseEnter,
    onMouseLeave,
}) => {
    const CLASSES_SIDEBAR = classNames({
        'collection-sidebar': true,
        [className]: true,
    });

    const sidebarItems = collection.items.map((item, index) => {
        const isActive = index === currentItemIndex;
        return (
            <CollectionSidebarItem
                key={`${isActive}_${item.itemId}_${item.title}`}
                onSelect={onSelectItem}
                selectItem={item.itemId}
                isActive={isActive}
                thumbnailURL= {item.thumbnails.small}
                title={item.title}
                subTitle={item.owner.displayName}
                index={index}
            />
        );
    });

    return (
        <div
            className={CLASSES_SIDEBAR}
            onMouseEnter={onMouseEnter}
            onMouseLeave={onMouseLeave}
        >
            <CollectionSidebarHeader
                title={collection.title}
                ownerURL={`https://twitch.tv/${collection.owner.name}`}
                ownerName={collection.owner.displayName}
                itemsLength={collection.items.length}
                totalDuration={collection.totalDuration}
                onClose={onClose}
            />
            <div className="collection-sidebar-item-container">
                {sidebarItems}
            </div>
        </div>
    );
};

CollectionSidebarComponent.propTypes = propTypes;
CollectionSidebarComponent.defaultProps = defaultProps;

export const AnimatedCollectionSidebar = makeAnimatedComponent(CollectionSidebarComponent, {
    showDuration: 350,
    hideDuration: 350,
    showClass: 'collection-sidebar--open',
    hideClass: 'collection-sidebar--close',
});
