import React from 'react';
import PropTypes from 'prop-types';
import { delay } from '../../../util/delay';
import pick from 'lodash/pick';
import classNames from 'classnames';

export const DEFAULT_OPTIONS = Object.freeze({
    showDuration: 0,
    hideDuration: 0,
    showClass: '',
    hideClass: '',
});

const TRANSITION_BUFFER = 100;

function getDisplayName(Component) {
    return Component.displayName || Component.name || 'Component';
}

/**
 * Wraps a Component in an AnimatedComponent class. Adds animation functionality
 * to the wrapped component, adding/removing classes based on the specified duration.
 * The wrapped component must accept className as a prop. The HOC applies animation
 * classes on the className prop.
 *
 * The animation classes in the options will be appended to the className prop
 * on the AnimatedComponent if it is defined.
 *
 * @param {Function|Object} Component: A React Component or a React Element
 * @return {Function} React Component
 */
export function makeAnimatedComponent(ComponentToWrap, opts = DEFAULT_OPTIONS) {
    const options = pick(opts, Object.keys(DEFAULT_OPTIONS));

    /**
     * onAnimationEnd {Func}
     *      - User supplied callback for when the component is fully unmounted.
     *
     * _transitionToNextComponent {Func}
     *      - Passed in when `AnimatedComponent` is wrapped with a `TransitionManager`,
     *        and is used to signal the `TransitionManager` when to render the next component
     */
    const propTypes = {
        onAnimationEnd: PropTypes.func,
        _transitionToNextComponent: PropTypes.func,
        className: PropTypes.string,
    };

    const defaultProps = {
        className: '',
    };

    class AnimatedComponent extends React.Component {
        constructor() {
            super(...arguments);
            this.state = {
                animationClass: '',
            };
        }

        _animateIn(done) {
            const { showDuration, showClass } = options;
            this.setState({
                animationClass: showClass,
            });

            delay(showDuration).
                then(done);
        }

        componentWillAppear(done) {
            this._animateIn(done);
        }

        componentDidAppear() {
            this.setState({ animationClass: '' });
        }

        componentWillEnter(done) {
            this._animateIn(done);
        }

        componentDidEnter() {
            this.setState({ animationClass: '' });
        }

        componentWillLeave(done) {
            const { hideDuration, hideClass } = options;
            this.setState({
                animationClass: hideClass,
            });

            delay(hideDuration - TRANSITION_BUFFER).
                then(done);
        }

        componentDidLeave() {
            if (this.props.onAnimationEnd) {
                this.props.onAnimationEnd();
            }

            if (this.props._transitionToNextComponent) {
                this.props._transitionToNextComponent();
            }
        }

        render() {
            const className = classNames(this.props.className, this.state.animationClass);

            return (
                <ComponentToWrap
                    {...this.props}
                    className={className}
                />
            );
        }
    }

    AnimatedComponent.propTypes = propTypes;
    AnimatedComponent.defaultProps = defaultProps;
    AnimatedComponent.displayName = `WrappedAnimatedComponent(${getDisplayName(ComponentToWrap)})`;

    return AnimatedComponent;
}
