import React from 'react';
import PropTypes from 'prop-types';
import { translate } from 'react-i18next';

const propTypes = {
    label: PropTypes.string.isRequired,
    onToggle: PropTypes.func,
    t: PropTypes.func.isRequired,
    toggledOn: PropTypes.bool,
};

const defaultProps = {
    onToggle() {},
    toggledOn: false,
};

function createOnEnterKeyFn(onClick) {
    return function onKeyUp({ key }) {
        if (key === 'Enter') {
            onClick();
        }
    };
}

export const ToggleComponent = ({ label, onToggle, t, toggledOn }) => {
    // http://stackoverflow.com/a/38872723
    const randomId = `toggle_button_${Math.random().toString(36).substr(2, 10)}`;

    /* Reasons for disabling lint rules:
     * jsx-a11y/no-static-element-interactions: ui-core uses a label, so we have to use a label here.
     *     Additionally, there is a requirement that the label also perform the toggle on click,
     *     so we have to make this interactive.
     *     * Note - eslint-disable-line was used here because eslint-disable-next-line did not work due to quirks
     *     in how eslint parses JSX. See https://github.com/eslint/eslint/issues/7030
     * jsx-a11y/no-noninteractive-element-to-interactive-role: This was around before the rule was added, and it
     *     is disabled to keep breakages minimal.
     */
    return (
        <div className="pl-player-menu__toggle-item">
            <div className="pl-toggle__label-wrapper">
                <label // eslint-disable-line jsx-a11y/no-static-element-interactions
                    className="pl-form__label qa-toggle-label"
                    htmlFor="toggle_button"
                    role="button" // eslint-disable-line jsx-a11y/no-noninteractive-element-to-interactive-role
                    onClick={onToggle}
                    onKeyUp={createOnEnterKeyFn(onToggle)}
                    tabIndex={0}
                >
                    {t(label)}
                </label>
            </div>
            <div className="pl-toggle">
                <input
                    className="qa-input-checkbox"
                    type="checkbox"
                    id={randomId}
                    checked={toggledOn}
                    onChange={onToggle}
                />
                <label
                    className="pl-toggle__button"
                    htmlFor={randomId}
                />
            </div>
        </div>
    );
};

ToggleComponent.propTypes = propTypes;
ToggleComponent.defaultProps = defaultProps;

export const Toggle = translate()(ToggleComponent);
