import React from 'react';
import PropTypes from 'prop-types';
import TransitionGroup from 'react-transition-group/TransitionGroup';

const propTypes = {
    children: PropTypes.node.isRequired,
    className: PropTypes.string,
    animate: PropTypes.bool.isRequired,
};

const defaultProps = {
    animate: false,
    className: '',
};

/**
 * This TransitionManager point wraps an AnimatedComponent, and allows multiple
 * AnimatedComponent to be rendered in series. If `animate` is false, the AnimatedComponent
 * is mounted/unmounted like a regular react component.
 */
export class TransitionManager extends React.Component {
    constructor() {
        super(...arguments);

        this.renderNextComponent = this.renderNextComponent.bind(this);

        this.state = {
            transitioning: false,
        };
    }

    componentWillReceiveProps() {
        const { animate } = this.props;
        if (animate) {
            this.setState({
                transitioning: true,
            });
        }
    }

    renderNextComponent() {
        this.setState({
            transitioning: false,
        });
    }

    createTransitionComponent() {
        const { children } = this.props;
        const child = React.Children.only(children);

        return React.cloneElement(child, {
            _transitionToNextComponent: this.renderNextComponent,
        });
    }

    render() {
        const { className, animate, children } = this.props;

        if (!animate) {
            return (
                <div className={className}>
                    {React.Children.only(children)}
                </div>
            );
        }

        const { transitioning } = this.state;
        const transitionComponent = !transitioning ? this.createTransitionComponent() : null;
        return (
            <TransitionGroup
                className={className}
                component={'div'}
            >
                {transitionComponent}
            </TransitionGroup>
        );
    }
}

TransitionManager.propTypes = propTypes;
TransitionManager.defaultProps = defaultProps;
