import React from 'react';
import { Interpolate, translate } from 'react-i18next';
import PropTypes from 'prop-types';
import { ClosePopupButton } from './buttons/close-popup-button';

/*
 * Based on the ExtensionProductPayload as defined at
 * https://git-aws.internal.justin.tv/gds/extension-coordinator/blob/master/src/api/extensions.ts#L73
 */
export const ProductPayloadPropTypes = PropTypes.shape({
    description: PropTypes.string.isRequired,
    developerName: PropTypes.string.isRequired,
    displayPrice: PropTypes.string.isRequired,
    extensionName: PropTypes.string.isRequired,
    shortDescription: PropTypes.string.isRequired,
    sku: PropTypes.string.isRequired,
    title: PropTypes.string.isRequired,
});

const propTypes = {
    cancel: PropTypes.func.isRequired,
    confirm: PropTypes.func.isRequired,
    productPayload: ProductPayloadPropTypes,
    isLinked: PropTypes.bool.isRequired,
    isLoggedIn: PropTypes.bool.isRequired,
    t: PropTypes.func.isRequired,
};

export function ExtensionsConfirmationComponent({
    cancel,
    confirm,
    isLinked,
    isLoggedIn,
    productPayload,
    t,
}) {
    if (!isLoggedIn) {
        return (
            <div className="extension-confirmation__wrapper">
                <div
                    className="
                        extension-confirmation__section
                        extension-confirmation__section--first
                        extension-confirmation__section--no-bottom

                    "
                >
                    <ClosePopupButton close={cancel} />
                </div>
                <div data-test-extensions-confirmation-logged-out={true} className="extension-confirmation__section">
                    <h4>{t('You must be logged in to use this extension feature.')}</h4>
                </div>
                <div className="extension-confirmation__section extension-confirmation__section--no-bottom">
                    <button
                        onClick={cancel}
                        className="extension-confirmation__cancel"
                        data-test-extension-confirmation-cancel={true}
                    >
                        {t('Close')}
                    </button>
                </div>
            </div>
        );
    }

    let linkNotice = null;
    if (!isLinked) {
        const notice = (
            'By clicking "Buy", you agree to share your Twitch ID ' +
            'with {{extensionName}}. You can manage account ' +
            'connections under'
        );
        linkNotice = (
            <div className="extension-confirmation__section extension-confirmation__section--no-bottom">
                <p data-test-extensions-confirmation-link-notice={true} className="extension-confirmation__link-notice">
                    <Interpolate
                        i18nKey={notice}
                        extensionName={productPayload.extensionName}
                    />{' '}
                    <a href="https://twitch.tv/settings/connections">{t('Settings')}</a>.
                </p>
            </div>
        );
    }

    const buttonSectionClassName = `extension-confirmation__section ${
        isLinked ? '' : 'extension-confirmation__section--no-bottom'
    }`;

    return (
        <div className="extension-confirmation__wrapper">
            <div
                className="
                    extension-confirmation__section
                    extension-confirmation__section--first
                    extension-confirmation__section--no-bottom
                "
            >
                <ClosePopupButton close={cancel} />
                <h4 className="extension-confirmation__popup-title">{t('In-Extension Purchase')}</h4>
            </div>
            <div className="extension-confirmation__section">
                <div>
                    <h5 className="extension-confirmation__product-title">
                        <span className="extension-confirmation__display-price">{productPayload.displayPrice}</span>
                        {productPayload.title}
                    </h5>
                    <p className="extension-confirmation__extension-name">
                        <Interpolate
                            i18nKey={'{{extensionName}} Extension'}
                            extensionName={productPayload.extensionName}
                        />
                    </p>
                </div>
                <p className="extension-confirmation__description">{productPayload.description}</p>
            </div>
            <div className={buttonSectionClassName}>
                <button
                    onClick={confirm}
                    className="extension-confirmation__purchase"
                    data-test-extension-confirmation-purchase={true}
                >
                    {t('Buy')}
                </button>
                <button
                    onClick={cancel}
                    className="extension-confirmation__cancel"
                    data-test-extension-confirmation-cancel={true}
                >
                    {t('Cancel')}
                </button>
            </div>
            {linkNotice}
        </div>
    );
}

ExtensionsConfirmationComponent.propTypes = propTypes;

export const ExtensionsConfirmation = translate()(ExtensionsConfirmationComponent);
