import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { translate, Interpolate } from 'react-i18next';
import { EXTENSION_PROP_TYPE } from 'state/extensions';
import { Button } from 'ui/components/buttons/base-button';
import { EXTENSION_MENU_DETAILS } from 'util/extensions';
import { twitchHost } from 'settings';

export const MENU_BACK_BUTTON = 'qa-menu-back-btn';
export const MENU_BACK_LABEL = 'qa-menu-back-label';
export const MENU_TOGGLE_IDENTITY_BUTTON = 'qa-menu-toggle-access-btn';
export const MENU_TOGGLE_IDENTITY_LABEL = 'qa-menu-toggle-access-label';
export const MENU_SUMMARY_LABEL = 'qa-menu-summary-label';
export const MENU_MANAGE_PERMISSION_LINK = 'qa-menu-manage-permission-link';

export const GRANT_ACCESS_HEADER = 'Grant Access';
export const REVOKE_ACCESS_HEADER = 'Revoke Access';

export const MANAGE_PERMISSIONS_TEXT = 'Manage Permissions';

export const GRANT_ACCESS_SUMMARY = 'You are granting {{extensionName}} the following access:';
export const REVOKE_ACCESS_SUMMARY = 'You are revoking the following access for {{extensionName}}:';

const TWITCH_ID_HEADER = 'Your Twitch User ID';
const TWITCH_ID_DESCRIPTION = 'Allows this extension to know your User ID on Twitch.';

const propTypes = {
    extension: EXTENSION_PROP_TYPE,
    onMenuTransition: PropTypes.func,
    onIdentityToggle: PropTypes.func,
    trackEvent: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
};

const defaultProps = {
    onMenuTransition() {},
    onIdentityToggle() {},
};

const CLASSES_SECTION = classNames(
    'pl-menu__section',
    'pl-menu__section--with-sep'
);

const CLASSES_HEADER_WITH_CARET = classNames(
    'pl-menu__item',
    'pl-menu__item--block',
    'pl-menu__item--with-caret-left'
);

const CLASSES_SUMMARY_LABEL = classNames(
    MENU_SUMMARY_LABEL,
    'pl-menu__item--label'
);

const CLASSES_ACCESS_DESC = classNames(
    'pl-menu__item--label',
    'extension-menu__access-list'
);

const CLASSES_ACCESS_HEADER = classNames(
    'extension-menu__access-list-header'
);

const CLASSES_ACCESS_FOOTER = classNames(
    'pl-menu__item--label',
    'extension-menu-footer'
);

const BUTTON_CLASSES = classNames(
    MENU_TOGGLE_IDENTITY_BUTTON,
    'pl-button',
    'extension-menu-footer__button'
);

export class ExtensionAccessMenuComponent extends React.Component {
    constructor() {
        super(...arguments);

        this.handleBackButtonClick = this.handleBackButtonClick.bind(this);
        this.handleIdentityToggleClick = this.handleIdentityToggleClick.bind(this);
    }

    handleBackButtonClick() {
        this.props.onMenuTransition(EXTENSION_MENU_DETAILS);
    }

    handleIdentityToggleClick() {
        this.props.onIdentityToggle(this.props.extension);

        const interaction = this.props.extension.lastUserIdentityLinkState ? 'revoke' : 'grant';
        this.props.trackEvent('extension_ui_interaction_client', this.props.extension, {
            // eslint-disable-next-line camelcase
            extension_interaction: interaction,
        });
    }

    render() {
        const { t, extension } = this.props;
        const settingsUrl = `${twitchHost}/settings/connections`;
        const nameComponent = <b>{extension.name}</b>;

        let headerLabel = GRANT_ACCESS_HEADER;
        let summaryLabel = (
            <Interpolate
                i18nKey={GRANT_ACCESS_SUMMARY}
                extensionName={nameComponent}
            />
        );

        if (extension.lastUserIdentityLinkState) {
            headerLabel = REVOKE_ACCESS_HEADER;
            summaryLabel = (
                <Interpolate
                    i18nKey={REVOKE_ACCESS_SUMMARY}
                    extensionName={nameComponent}
                />
            );
        }

        return (
            <div className="pl-menu__inner">
                <div className={CLASSES_HEADER_WITH_CARET}>
                    <Button
                        onClick={this.handleBackButtonClick}
                        className={MENU_BACK_BUTTON}
                    >
                        <span className={MENU_BACK_LABEL}>{t(headerLabel)}</span>
                    </Button>
                </div>
                <div className={CLASSES_SECTION}>
                    <div className={CLASSES_SUMMARY_LABEL}>
                        <span>{summaryLabel}</span>
                    </div>
                    <div className={CLASSES_ACCESS_DESC}>
                        <div className={CLASSES_ACCESS_HEADER}>{t(TWITCH_ID_HEADER)}</div>
                        <span>{t(TWITCH_ID_DESCRIPTION)}</span>
                        <p>
                            <a
                                href={settingsUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                className={MENU_MANAGE_PERMISSION_LINK}
                            >
                                {t(MANAGE_PERMISSIONS_TEXT)}
                            </a>
                        </p>
                    </div>

                    <div className={CLASSES_ACCESS_FOOTER}>
                        <Button
                            onClick={this.handleIdentityToggleClick}
                            className={BUTTON_CLASSES}
                        >
                            <span className={MENU_TOGGLE_IDENTITY_LABEL}>{t(headerLabel)}</span>
                        </Button>
                    </div>

                </div>
            </div>
        );
    }
}

ExtensionAccessMenuComponent.propTypes = propTypes;
ExtensionAccessMenuComponent.defaultProps = defaultProps;

export const ExtensionAccessMenu = translate()(ExtensionAccessMenuComponent);
