import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { translate } from 'react-i18next';
import { EXTENSION_PROP_TYPE } from 'state/extensions';
import { Button } from 'ui/components/buttons/base-button';
import {
    EXTENSION_MENU_MAIN,
    EXTENSION_MENU_MANAGE_ACCESS,
    EXTENSION_MENU_REPORT,
    EXTENSION_PERMISSION_STATE_GRANTED,
    EXTENSION_ROLE_BROADCASTER,
} from 'util/extensions';

const VIEW_DETAILS = 'View Details';
const REPORT_EXTENSION = 'Report Extension';

const MANAGE_ACCESS_FULL = 'Manage Access';
const MANAGE_ACCESS_SHORT = 'Access';
const ACCESS_GRANTED = 'Granted';
const ACCESS_REQUESTED = '!';

const EXT_DETAILS_TARGET = '_blank';
export const TWITCH_EXT_DETAILS_BASE = 'https://www.twitch.tv/ext';

const propTypes = {
    extension: EXTENSION_PROP_TYPE,
    isLoggedIn: PropTypes.bool.isRequired,
    showBackButton: PropTypes.bool,
    onMenuTransition: PropTypes.func,
    trackEvent: PropTypes.func.isRequired,
    windowObj: PropTypes.object.isRequired,
    t: PropTypes.func.isRequired,
};

const defaultProps = {
    onMenuTransition() {},
    showBackButton: true,
};

const CLASSES_LIST_HEADER = classNames({
    'pl-menu__section': true,
    'pl-menu__item--label': true,
    'extension-menu-details__header': true,
});

const CLASSES_ITEM_SUMMARY = classNames({
    'pl-menu__item--label': true,
    'extension-menu-details__desc': true,
});

const CLASSES_LIST_HEADER_WITH_CARET = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret-left': true,
});

const CLASSES_ITEM = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
});

const CLASSES_ITEM_WITH_CARET = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret': true,
});

const CLASSES_LIST_SECTION = classNames({
    'pl-menu__section': true,
    'pl-menu__section--with-sep': true,
});

export const EXT_BACK_BUTTON = 'qa-extension-back-button';
export const EXT_REPORT_BUTTON = 'qa-extension-report-button';
export const EXT_DETAILS_BUTTON = 'qa-extension-details-button';
export const EXT_MANAGE_ACCESS_BUTTON = 'qa-extension-manage-access-button';

export class ExtensionDetailsMenuComponent extends React.Component {
    constructor() {
        super(...arguments);

        this.handleBackButtonClick = this.handleBackButtonClick.bind(this);
        this.handleViewDetailsClick = this.handleViewDetailsClick.bind(this);
        this.handleManageAccessClick = this.handleManageAccessClick.bind(this);
        this.handleReportExtensionClick = this.handleReportExtensionClick.bind(this);
    }

    handleBackButtonClick() {
        this.props.onMenuTransition(EXTENSION_MENU_MAIN);
    }

    handleViewDetailsClick() {
        const detailsUrl = `${TWITCH_EXT_DETAILS_BASE}/${this.props.extension.id}-${this.props.extension.version}`;
        this.props.windowObj.open(detailsUrl, EXT_DETAILS_TARGET);
    }

    handleManageAccessClick() {
        this.props.onMenuTransition(EXTENSION_MENU_MANAGE_ACCESS);
    }

    handleReportExtensionClick() {
        this.props.onMenuTransition(EXTENSION_MENU_REPORT);
        this.props.trackEvent('extension_ui_interaction_client', this.props.extension, {
            // eslint-disable-next-line camelcase
            extension_interaction: 'report',
        });
    }

    render() {
        const { extension, t } = this.props;
        const backButton = this._renderHeader();
        const viewDetailsButton = (
            <div className={CLASSES_ITEM}>
                <Button
                    onClick={this.handleViewDetailsClick}
                    className={EXT_DETAILS_BUTTON}
                >
                    {t(VIEW_DETAILS)}
                </Button>
            </div>
        );

        const manageAccessButton = this._renderManageAccess();

        const reportExtensionButton = (
            <div className={CLASSES_ITEM_WITH_CARET}>
                <Button
                    onClick={this.handleReportExtensionClick}
                    className={EXT_REPORT_BUTTON}
                >
                    {t(REPORT_EXTENSION)}
                </Button>
            </div>
        );

        return (
            <div className="pl-menu__inner">
                {backButton}
                <div className={CLASSES_LIST_SECTION}>
                    <div className={CLASSES_ITEM_SUMMARY}>
                        {extension.summary}
                    </div>
                    {viewDetailsButton}
                    {manageAccessButton}
                    {reportExtensionButton}
                </div>
            </div>
        );
    }

    _renderHeader() {
        const { extension, showBackButton } = this.props;
        if (showBackButton) {
            return (
                <div className={CLASSES_LIST_HEADER_WITH_CARET}>
                    <Button
                        onClick={this.handleBackButtonClick}
                        className={EXT_BACK_BUTTON}
                    >
                        {extension.name}
                    </Button>
                </div>
            );
        }

        return (
            <div className={CLASSES_LIST_HEADER}>
                {extension.name}
            </div>
        );
    }

    _renderManageAccess() {
        const { extension, t } = this.props;
        const extensionSupportsIdentityLinking = extension.supportsIdentityLinking;
        const extensionIsLinked = extension.token.permissionsState === EXTENSION_PERMISSION_STATE_GRANTED;
        const shouldShowUserPermissions = extension.token.role !== EXTENSION_ROLE_BROADCASTER &&
            this.props.isLoggedIn &&
            (extensionSupportsIdentityLinking || extensionIsLinked);

        if (!shouldShowUserPermissions) {
            return null;
        }

        const accessStatusClassName = classNames({
            'pl-pill': true,
            'extension-access-granted': extension.lastUserIdentityLinkState,
            'extension-access-requested': !extension.lastUserIdentityLinkState,
        });

        let accessLabel = MANAGE_ACCESS_FULL;
        let accessStatusLabel = ACCESS_REQUESTED;

        if (extension.lastUserIdentityLinkState) {
            accessLabel = MANAGE_ACCESS_SHORT;
            accessStatusLabel = t(ACCESS_GRANTED);
        }

        return (
            <div className={CLASSES_ITEM_WITH_CARET}>
                <Button
                    onClick={this.handleManageAccessClick}
                    className={EXT_MANAGE_ACCESS_BUTTON}
                >
                    <span>{t(accessLabel)}</span>&nbsp;
                    <span className={accessStatusClassName}>{accessStatusLabel}</span>
                </Button>
            </div>
        );
    }
}

ExtensionDetailsMenuComponent.propTypes = propTypes;
ExtensionDetailsMenuComponent.defaultProps = defaultProps;

export const ExtensionDetailsMenu = translate()(ExtensionDetailsMenuComponent);
