import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { translate, Interpolate } from 'react-i18next';
import { Button } from 'ui/components/buttons/base-button';
import { Toggle } from 'ui/components/common/toggle';
import { EXTENSION_PROP_TYPE } from 'state/extensions';

export const MODAL_TITLE_HEADER = 'qa-follow-modal-title-header';
export const MODAL_TITLE_CHANNEL = 'qa-follow-modal-title-channel';
export const MODAL_BUTTON_CONFIRM = 'qa-follow-modal-button-confirm';
export const MODAL_BUTTON_CANCEL = 'qa-follow-modal-button-cancel';
export const MODAL_NOTIFICATION_TOGGLE = 'qa-notification-toggle';

const CLASSES_MODAL_INNER = classNames({
    'pl-menu__inner': true,
    'extension-modal__inner': true,
});

const CLASSES_MODAL_HEADER = classNames({
    'pl-menu__section': true,
    'extension-modal__header': true,
});

const CLASSES_MODAL_TITLE = classNames({
    [MODAL_TITLE_HEADER]: true,
    'pl-menu__item': true,
    'extension-modal__title': true,
});

const CLASSES_MODAL_TITLE_CHANNEL = classNames({
    [MODAL_TITLE_CHANNEL]: true,
    'pl-menu__item--label': true,
    'extension-modal__title-channel': true,
});

const CLASSES_LIST_SECTION = classNames({
    'pl-menu__section': true,
    'pl-menu__section--with-sep': true,
});

const CLASSES_INFO_TEXT = classNames({
    'pl-menu__item--label': true,
    'extension-modal__info-text': true,
});

const CLASSES_FOLLOW_BUTTON = classNames({
    [MODAL_BUTTON_CONFIRM]: true,
    'pl-button': true,
    'extension-modal__button': true,
});

const CLASSES_CANCEL_BUTTON = classNames({
    [MODAL_BUTTON_CANCEL]: true,
    'pl-button': true,
    'pl-button--hollow': true,
    'extension-modal__button': true,
});

const CLASSES_CLOSE_BUTTON = classNames({
    'pl-close-button': true,
    'extension-modal__close-button': true,
});

export const EXTENSION_FOLLOW_REQUEST_TYPE = PropTypes.shape({
    channel: PropTypes.string.isRequired,
    isFollowing: PropTypes.bool.isRequired,
});

const propTypes = {
    t: PropTypes.func.isRequired,
    extension: EXTENSION_PROP_TYPE,
    options: EXTENSION_FOLLOW_REQUEST_TYPE.isRequired,
    onClose: PropTypes.func.isRequired,
};

// eslint-disable-next-line max-len
const FOLLOW_INFO_TEXT = 'Receive updates on this channel\'s activity, including when they go live and upload videos or clips, on mobile, email, and web.';
const NOTIFICATION_TEXT = 'Notifications';

export class ExtensionFollowModalComponent extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            notifications: false,
        };

        this.confirmFollow = this._confirmFollow.bind(this);
        this.cancelFollow = this._cancelFollow.bind(this);
        this.onNotificationToggle = this._onNotificationToggle.bind(this);
    }

    render() {
        return (
            <div className={CLASSES_MODAL_INNER}>
                {this._renderHeader()}
                <div className={CLASSES_LIST_SECTION}>
                    {this._renderButtons()}
                </div>
                { this._renderNotificationToggle() }
            </div>
        );
    }

    _renderHeader() {
        const { channel, isFollowing }  = this.props.options;
        let headerText;

        if (!channel) {
            headerText = 'The channel you are attempting to follow does not exist';
        } else if (isFollowing) {
            headerText = 'You are already following {{channelName}}';
        } else {
            headerText = 'You are about to follow {{channelName}}';
        }

        let headerComponent;

        if (channel) {
            const channelTitleComponent = (
                <span className={CLASSES_MODAL_TITLE_CHANNEL}>
                    {channel}
                </span>
            );

            headerComponent = (
                <Interpolate
                    i18nKey={headerText}
                    channelName={channelTitleComponent}
                />
            );
        } else {
            headerComponent = (
                <div className="extension-modal__title-info">
                    {this.props.t(headerText)}
                </div>
            );
        }

        return (
            <div className={CLASSES_MODAL_HEADER}>
                <div className="extension-modal__header-row">
                    <div className={CLASSES_MODAL_TITLE}>
                        { headerComponent }
                    </div>
                    <Button className={CLASSES_CLOSE_BUTTON} onClick={this.cancelFollow}>
                        <svg className="pl-close-button__svg">
                            <use xlinkHref="#pl-icon_close" />
                        </svg>
                    </Button>
                </div>
            </div>
        );
    }

    _renderButtons() {
        const { channel, isFollowing } = this.props.options;

        if (!channel || isFollowing) {
            return (
                <Button className={CLASSES_CANCEL_BUTTON} onClick={this.cancelFollow}>
                    {this.props.t('Close')}
                </Button>
            );
        }

        return [
            <Button key="confirm" className={CLASSES_FOLLOW_BUTTON} onClick={this.confirmFollow}>
                {this.props.t('Follow')}
            </Button>,
            <Button key="cancel" className={CLASSES_CANCEL_BUTTON} onClick={this.cancelFollow}>
                {this.props.t('Cancel')}
            </Button>,
        ];
    }

    _renderNotificationToggle() {
        const { channel, isFollowing } = this.props.options;
        const { notifications } = this.state;

        if (!channel || isFollowing) {
            return null;
        }

        return (
            <div className={CLASSES_LIST_SECTION}>
                <Toggle
                    className={MODAL_NOTIFICATION_TOGGLE}
                    label={NOTIFICATION_TEXT}
                    onToggle={this.onNotificationToggle}
                    toggledOn={notifications}
                />
                <div className={CLASSES_INFO_TEXT}>
                    {this.props.t(FOLLOW_INFO_TEXT)}
                </div>
            </div>
        );
    }

    _confirmFollow() {
        this.props.onClose({
            didFollow: true,
            notifications: this.state.notifications,
        });
    }

    _cancelFollow() {
        this.props.onClose({
            didFollow: false,
            notifications: false,
        });
    }

    _onNotificationToggle() {
        this.setState({ notifications: !this.state.notifications });
    }
}

ExtensionFollowModalComponent.propTypes = propTypes;

export const ExtensionFollowModal = translate()(ExtensionFollowModalComponent);
