import React from 'react';
import PropTypes from 'prop-types';
import { ExtensionCoordinator } from 'extension-coordinator';
import { Dobbin } from 'dobbin';
import { EXTENSION_PERMISSION_STATE_GRANTED } from 'util/extensions';

export const EXT_OVERLAY_CLASS = 'extension-overlay';
export const EXT_IFRAME_CLASS = 'extension-overlay__iframe';

const propTypes = {
    onDoubleClick: PropTypes.func.isRequired,
    width: PropTypes.number.isRequired,
    height: PropTypes.number.isRequired,
    extension: PropTypes.shape({
        id: PropTypes.string.isRequired,
        name: PropTypes.string.isRequired,
        sku: PropTypes.string.isRequired,
        anchor: PropTypes.string.isRequired,
        vendorCode: PropTypes.string.isRequired,
        version: PropTypes.string.isRequired,
        viewerUrl: PropTypes.string.isRequired,
        state: PropTypes.string.isRequired,
        lastUserIdentityLinkState: PropTypes.bool.isRequired,
        supportsIdentityLinking: PropTypes.bool.isRequired,
        token: PropTypes.shape({
            token: PropTypes.string.isRequired,
            permissionsState: PropTypes.string.isRequired,
        }),
    }).isRequired,
    game: PropTypes.string.isRequired,
    isLoggedIn: PropTypes.bool.isRequired,
    language: PropTypes.string.isRequired,
    locale: PropTypes.string.isRequired,
    loginId: PropTypes.number.isRequired,
    onIdentityLinked: PropTypes.func.isRequired,
    onModalRequested: PropTypes.func.isRequired,
    trackingProperties: PropTypes.shape({
        login: PropTypes.string.isRequired,
        channel: PropTypes.string.isRequired,
        channelId: PropTypes.number.isRequired,
        deviceId: PropTypes.string.isRequired,
        platform: PropTypes.string.isRequired,
        playerType: PropTypes.string.isRequired,
    }),
    onBeginPurchase: PropTypes.func,
};

export class ExtensionOverlay extends React.Component {
    constructor(props) {
        super(props);

        this._boundOnFrameDoubleClick = this._onFrameDoubleClick.bind(this);
        this._boundIframeHostRefHandler = this._iframeHostRefHandler.bind(this);
        this._boundOnIdentityLinked = this._onIdentityLinked.bind(this);
        this._boundOnModalRequested = this._onModalRequested.bind(this);
    }

    componentDidMount() {
        const { isLoggedIn } = this.props;
        const loginId = isLoggedIn ? this.props.loginId : 0;

        this.extensionFrame = new ExtensionCoordinator.ExtensionFrame({
            dobbin: Dobbin,
            extension: {
                id: this.props.extension.id,
                anchor: this.props.extension.anchor,
                clientId: this.props.extension.id,
                name: this.props.extension.name,
                sku: this.props.extension.sku,
                token: this.props.extension.token.token,
                version: this.props.extension.version,
                state: this.props.extension.state,
                vendorCode: this.props.extension.vendorCode,
                viewerUrl: this.props.extension.viewerUrl,
                requestIdentityLink: this.props.extension.supportsIdentityLinking,
            },
            mode: 'viewer',
            language: this.props.language,
            locale: this.props.locale,
            parentElement: this.parentElement,
            iframeClassName: EXT_IFRAME_CLASS,
            loginId,
            channelId: this.props.trackingProperties.channelId,
            trackingProperties: {
                /* eslint-disable camelcase */
                login: this.props.trackingProperties.login,
                channel: this.props.trackingProperties.channel,
                device_id: this.props.trackingProperties.deviceId,
                platform: this.props.trackingProperties.platform,
                player_type: this.props.trackingProperties.playerType,
                /* eslint-enable camelcase */
            },
            onBeginPurchase: this.props.onBeginPurchase,
        });

        this.parentElement.addEventListener('dblclick', this._boundOnFrameDoubleClick);
        this.extensionFrame.on('identityLinked', this._boundOnIdentityLinked);
        this.extensionFrame.on('requestModal', this._boundOnModalRequested);
        ExtensionCoordinator.ExtensionService.postContext({
            game: this.props.game,
        });
    }

    componentWillUnmount() {
        this.parentElement.removeEventListener('dblclick', this._boundOnFrameDoubleClick);
        this.extensionFrame.off('identityLinked', this._boundOnIdentityLinked);
        this.extensionFrame.destroy();
    }

    componentDidUpdate() {
        const isLinked = this.props.extension.token.permissionsState === EXTENSION_PERMISSION_STATE_GRANTED;
        const linkRequested = this.props.extension.lastUserIdentityLinkState;

        if (linkRequested !== isLinked) {
            if (linkRequested) {
                this.extensionFrame.linkIdentity();
            } else {
                this.extensionFrame.unlinkIdentity();
            }
        }
    }

    render() {
        const divStyle = {
            width: `${this.props.width}px`,
            height: `${this.props.height}px`,
        };
        return (
            <div
                className={EXT_OVERLAY_CLASS}
                ref={this._boundIframeHostRefHandler}
                style={divStyle}
            />
        );
    }

    _onIdentityLinked(isLinked) {
        const { extension, onIdentityLinked } = this.props;
        if (!onIdentityLinked) {
            return;
        }

        onIdentityLinked(extension.id, isLinked);
    }

    _onFrameDoubleClick(evt) {
        evt.preventDefault();
        this.props.onDoubleClick();
    }

    _onModalRequested(confirmationRequest) {
        this.props.onModalRequested(confirmationRequest);
    }

    _iframeHostRefHandler(element) {
        this.parentElement = element;
    }
}

ExtensionOverlay.propTypes = propTypes;
