import PropTypes from 'prop-types';
import React from 'react';
import classNames from 'classnames';
import assign from 'lodash/assign';
import pick from 'lodash/pick';
import { translate, Interpolate } from 'react-i18next';
import { Button } from 'ui/components/buttons/base-button';
import { EXTENSION_MENU_DETAILS } from 'util/extensions';
import { postExtensionReport } from 'api';

const propTypes = {
    t: PropTypes.func.isRequired,
    channel: PropTypes.string.isRequired,
    onMenuTransition: PropTypes.func,
    authenticityToken: PropTypes.string.isRequired,
    extension: PropTypes.shape({
        id: PropTypes.string.isRequired,
        name: PropTypes.string.isRequired,
        version: PropTypes.string.isRequired,
    }).isRequired,
};

const defaultProps = {
    onMenuTransition() {},
};

export const MENU_BACK_BUTTON = 'qa-menu-back-btn';
export const MENU_BACK_LABEL = 'qa-menu-back-label';
export const EXT_SUBMIT_BUTTON = 'qa-ext-report-submit-btn';
export const EXT_SUBMIT_LABEL = 'qa-ext-report-submit-label';

const REPORT_EXTENSION = 'Report Extension';
const TOS_LINK_TEXT = 'terms of service';
const REPORT_DESCRIPTION_PLACEHOLDER = 'Description (required)';
const EXT_REPORT_SENT = 'Sent!';
const EXT_REPORT_SUBMIT = 'Submit';

// eslint-disable-next-line max-len
const TOS_HEADER_TEXT = 'Tell us why there has been a violation of the {{tosLink}}. (Abuse of this feature is violation of the terms of service.)';

const REPORT_INPUT_CLASSES = classNames(
    'pl-form__input',
    'extension-report__desc'
);

const CLASSES_ACCESS_FOOTER = classNames({
    'extension-menu-footer': true,
});

const CLASSES_LIST_HEADER = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret-left': true,
});

const CLASSES_LIST_SECTION = classNames({
    'pl-menu__section': true,
    'pl-menu__section--with-sep': true,
});

const CLASSES_ITEM_SUMMARY = classNames({
    'pl-mg-b-1': true,
    'pl-menu__item': true,
    'pl-menu__item--label': true,
});

const CLASSES_ITEM = classNames({
    'pl-menu__item': true,
    'pl-menu__item--label': true,
});

const EXTENSION_REPORT_FIELDS = [
    'reported_user',
    'authenticity_token',
    'tos_ban',
    'reason',
    'description',
];

export class ExtensionReportMenuComponent extends React.Component {
    constructor(props) {
        super(...arguments);

        this.state = {
            /* eslint-disable camelcase */
            reported_user: props.channel,
            authenticity_token: props.authenticityToken,
            tos_ban: 'false',
            /* eslint-enable camelcase */
            reason: '',
            description: '',
            sent: false,
        };
        this.canSubmit = this.canSubmit.bind(this);
        this.handleSubmit = this.handleSubmit.bind(this);
        this.handleInputChange = this.handleInputChange.bind(this);
        this.handleBackButtonClick = this.handleBackButtonClick.bind(this);
    }

    canSubmit() {
        const { sent, description, reason } = this.state;
        const submittable = description && reason;
        return sent ? false : submittable;
    }

    handleBackButtonClick() {
        this.props.onMenuTransition(EXTENSION_MENU_DETAILS);
    }

    handleInputChange(event) {
        this.setState({
            [event.target.name]: event.target.value,
        });
    }

    handleSubmit() {
        // add extensions data to description
        const description = [
            this.state.description,
            `Extension ID: ${this.props.extension.id}`,
            `Extension Name: ${this.props.extension.name}`,
            `Extension Version: ${this.props.extension.version}`,
        ].join('\n');

        return postExtensionReport(pick(assign({}, this.state, { description }), EXTENSION_REPORT_FIELDS)).then(() => {
            this.setState({
                sent: true,
            });
        });
    }

    render() {
        const tosLink = (
            <a href="https://www.twitch.tv/p/terms-of-service" target="_blank" rel="noopener noreferrer">
                {this.props.t(TOS_LINK_TEXT)}
            </a>
        );
        const canSubmit = this.canSubmit();
        const submitButtonClasses = classNames({
            [EXT_SUBMIT_BUTTON]: true,
            'pl-button': true,
            'pl-button--hollow': !canSubmit,
        });

        const formElements = (
            <form>
                <select
                    className={REPORT_INPUT_CLASSES}
                    name="reason"
                    value={this.state.reason}
                    onBlur={this.handleInputChange}
                    onChange={this.handleInputChange}
                >
                    <option value="">Select Reason</option>
                    <option value="selfharm">Self-Harm</option>
                    <option value="harm">Attempts or Threats to Harm</option>
                    <option value="harassment">Harassment</option>
                    <option value="hate_speech">Hate Speech</option>
                    <option value="impersonation">Impersonation</option>
                    <option value="spam">Spam, Scams, or Other Malicious Content</option>
                    <option value="porn">Porn or Other Sexually Explicit Content</option>
                    <option value="nudity">Nudity or Sexual Behavior/Attire</option>
                    <option value="gore">Extreme Violence, Gore, or Other Obscene Content</option>
                    <option value="cheating">Cheating in Online Game</option>
                    <option value="bits_violation">Bits Acceptable Use Policy Violation</option>
                    <option value="other">Other Terms of Service Violation</option>
                </select>
                <br />
                <textarea
                    className={REPORT_INPUT_CLASSES}
                    name="description"
                    value={this.state.description}
                    rows={2}
                    onChange={this.handleInputChange}
                    placeholder={this.props.t(REPORT_DESCRIPTION_PLACEHOLDER)}
                />
                <div className={CLASSES_ACCESS_FOOTER}>
                    <Button
                        disabled={!canSubmit}
                        className={submitButtonClasses}
                        onClick={this.handleSubmit}
                    >
                        <span
                            className={EXT_SUBMIT_LABEL}
                        >
                            {this.props.t(this.state.sent ? EXT_REPORT_SENT : EXT_REPORT_SUBMIT)}
                        </span>
                    </Button>
                </div>
            </form>
        );

        return (
            <div className="pl-menu__inner">
                <div className={CLASSES_LIST_HEADER}>
                    <Button
                        onClick={this.handleBackButtonClick}
                        className={MENU_BACK_BUTTON}
                    >
                        <span
                            className={MENU_BACK_LABEL}
                        >
                            {this.props.t(REPORT_EXTENSION)}
                        </span>
                    </Button>
                </div>
                <div className={CLASSES_LIST_SECTION}>
                    <div className={CLASSES_ITEM_SUMMARY}>
                        <Interpolate
                            i18nKey={TOS_HEADER_TEXT}
                            tosLink={tosLink}
                        />
                    </div>
                    <div className={CLASSES_ITEM}>
                        {formElements}
                    </div>
                </div>
            </div>
        );
    }
}

ExtensionReportMenuComponent.propTypes = propTypes;
ExtensionReportMenuComponent.defaultProps = defaultProps;

export const ExtensionReportMenu = translate()(ExtensionReportMenuComponent);
