import React from 'react';
import PropTypes from 'prop-types';
import { SETTINGS_INNER_PADDING } from '../settings/menu-manager';
import { EXTENSION_PROP_TYPE } from 'state/extensions';
import { ExtensionListMenu } from 'ui/components/extensions/extension-list-menu';
import { ExtensionDetailsMenu } from 'ui/components/extensions/extension-details-menu';
import { ExtensionAccessMenu } from 'ui/components/extensions/extension-access-menu';
import { ExtensionReportMenu } from 'ui/components/extensions/extension-report-menu';
import {
    EXTENSION_MENU_MAIN,
    EXTENSION_MENU_DETAILS,
    EXTENSION_MENU_MANAGE_ACCESS,
    EXTENSION_MENU_REPORT,
} from 'util/extensions';

const propTypes = {
    playerHeight: PropTypes.number.isRequired,
    onIdentityToggle: PropTypes.func.isRequired,
    extensions: PropTypes.arrayOf(EXTENSION_PROP_TYPE).isRequired,
    isLoggedIn: PropTypes.bool.isRequired,
    trackEvent: PropTypes.func.isRequired,
    windowObj: PropTypes.object.isRequired,
    submenu: PropTypes.string,
    handleMenuTransition: PropTypes.func.isRequired,
};

export class ExtensionsMenuManager extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            currentExtension: this.props.extensions[0],
        };

        this.handleExtensionClick = this.handleExtensionClick.bind(this);
    }

    handleExtensionClick(extension) {
        this.props.handleMenuTransition(EXTENSION_MENU_DETAILS);
        this.setState({
            currentExtension: extension,
        });
        this.props.trackEvent('extension_setting_click', extension);
    }

    componentDidMount() {
        if (this.props.extensions.length === 1 && !this.props.submenu) {
            this.handleExtensionClick(this.props.extensions[0]);
        }
    }

    render() {
        const { playerHeight } = this.props;
        const currentMenu = this._renderCurrentMenu();
        const settingsStyle = {
            maxHeight: `${playerHeight - SETTINGS_INNER_PADDING}px`,
        };

        return (
            <div
                className="pl-menu"
                style={settingsStyle}
            >
                {currentMenu}
            </div>
        );
    }

    _renderCurrentMenu() {
        switch (this.props.submenu) {
        case EXTENSION_MENU_DETAILS:
            return (
                <ExtensionDetailsMenu
                    onMenuTransition={this.props.handleMenuTransition}
                    showBackButton={this.props.extensions.length > 1}
                    extension={this.state.currentExtension}
                    isLoggedIn={this.props.isLoggedIn}
                    trackEvent={this.props.trackEvent}
                    windowObj={this.props.windowObj}
                />
            );
        case EXTENSION_MENU_MANAGE_ACCESS:
            return (
                <ExtensionAccessMenu
                    onMenuTransition={this.props.handleMenuTransition}
                    onIdentityToggle={this.props.onIdentityToggle}
                    trackEvent={this.props.trackEvent}
                    extension={this.state.currentExtension}
                />
            );
        case EXTENSION_MENU_REPORT:
            return (
                <ExtensionReportMenu
                    onMenuTransition={this.props.handleMenuTransition}
                    channel={this.state.channel}
                    extension={this.state.currentExtension}
                />
            );
        case EXTENSION_MENU_MAIN:
        default:
            return (
                <ExtensionListMenu
                    onExtensionClick={this.handleExtensionClick}
                    extensions={this.props.extensions}
                />
            );
        }
    }
}

ExtensionsMenuManager.propTypes = propTypes;
