import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { EXTENSION_PROP_TYPE } from 'state/extensions';
import { ExtensionFollowModal } from './extension-follow-modal';
import { ExtensionFeatureNeedsAuthModal } from './extension-feature-needs-auth-modal';

export const MODAL_BACKDROP = 'qa-modal-backdrop';

const CLASSES_EXTENSION_MODAL_BACKDROP = classNames(
    'extension-overlays',
    'extension-modal__backdrop',
    MODAL_BACKDROP
);

const CLASSES_EXTENSION_MODAL = classNames({
    'pl-menu': true,
    'extension-modal__main': true,
});

const EXTENSION_MODAL_NONE = 'no modal';
const EXTENSION_MODAL_FOLLOW = 'follow modal';
const EXTENSION_MODAL_NEED_AUTH = 'needs auth modal';

const EXTENSION_ACTION_MAP = {
    'twitch-ext-follow-action': EXTENSION_MODAL_FOLLOW,
    'twitch-ext-login-request': EXTENSION_MODAL_NEED_AUTH,
};

const propTypes = {
    playerHeight: PropTypes.number.isRequired,
    extensions: PropTypes.arrayOf(EXTENSION_PROP_TYPE).isRequired,
    modalRequest: PropTypes.shape({
        action: PropTypes.string.isRequired,
        options: PropTypes.object,
        resultCallback: PropTypes.func.isRequired,
        defaultResult: PropTypes.object.isRequired,
    }),
    onModalClose: PropTypes.func.isRequired,
};

export class ExtensionsModalManager extends React.Component {
    constructor() {
        super(...arguments);

        this.onModalClose = this._onModalClose.bind(this);
        this.cancelModal = this._cancelModal.bind(this);
    }

    get currentModal() {
        if (!(this.props.modalRequest && this.props.modalRequest.action)) {
            return EXTENSION_MODAL_NONE;
        }

        return EXTENSION_ACTION_MAP[this.props.modalRequest.action];
    }

    render() {
        if (this.currentModal === EXTENSION_MODAL_NONE) {
            return null;
        }

        const { playerHeight } = this.props;

        const modalStyle = {
            maxHeight: `${playerHeight}px`,
        };

        return (
            // eslint-disable-next-line jsx-a11y/no-static-element-interactions, jsx-a11y/click-events-have-key-events
            <div
                role="button"
                tabIndex={-1}
                className={CLASSES_EXTENSION_MODAL_BACKDROP}
                onClick={this.cancelModal}
            >
                <div
                    className={CLASSES_EXTENSION_MODAL}
                    style={modalStyle}
                >
                    {this._renderCurrentModal()}
                </div>
            </div>
        );
    }

    _renderCurrentModal() {
        switch (this.currentModal) {
        case EXTENSION_MODAL_FOLLOW:
            return (
                <ExtensionFollowModal
                    extension={this.props.extensions[0]}
                    options={this.props.modalRequest.options}
                    onClose={this.onModalClose}
                />
            );
        case EXTENSION_MODAL_NEED_AUTH:
            return (
                <ExtensionFeatureNeedsAuthModal
                    onClose={this.cancelModal}
                />
            );
        case EXTENSION_MODAL_NONE:
        default:
            return null;
        }
    }

    _onModalClose(result) {
        const { resultCallback } = this.props.modalRequest;

        if (resultCallback) {
            resultCallback(result);
        }

        this.props.onModalClose();
    }

    _cancelModal(event) {
        if (event && event.target !== event.currentTarget) {
            return;
        }

        this.onModalClose(this.props.modalRequest.defaultResult);
    }
}

ExtensionsModalManager.propTypes = propTypes;
