import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';

import assign from 'lodash/assign';
import isObject from 'lodash/isObject';
import range from 'lodash/range';
import { translate } from 'react-i18next';

import { Button } from 'ui/components/buttons/base-button';
import {
    MONTH,
    getMaxDays,
    isMinimumAge,
} from 'util/date';
import { sessionStore } from 'util/storage';

const AGE_GATE_SUBMIT_LABEL = 'Submit';
export const KEY_AGE_GATES_FAILED = 'age_gates_failed';
export const KEY_AGE_GATE = 'age_gate';

const AGE_RESTRICTION_OVERLAY = classNames(
    'player-overlay',
    'pl-age-restriction-overlay--wall',
    'pl-age-restriction-overlay',
    'js-age-restriction-overlay'
);

const PLAYER_CENTER_CONTENT = classNames(
    'player-center-content'
);

const AGE_GATE_CLASS = classNames(
    'pl-age-gate js-age-gate'
);

const AGE_GATE_SUBMIT_BUTTON_CLASS = classNames(
    'js-age-gate-submit player-content-button'
);

const DAY_DATEPICKER_CLASS = classNames(
    'player-datepicker--select js-select-day'
);

const MONTH_DATEPICKER_CLASS = classNames(
    'player-datepicker--select js-select-month'
);

const YEAR_DATEPICKER_CLASS = classNames(
    'player-datepicker--select js-select-year'
);

const AGE_GATE_FAILED_ONCE_CLASS = classNames(
    'player-age-gate-failed-once'
);

const AGE_GATE_FAILED_ICON_CLASS = classNames(
    'player-age-gate--fail-icon'
);

const AGE_GATE_LOCKED_OUT_LABEL_CLASS = classNames(
    'age-gate-locked-out-label'
);

const AGE_GATE_WARNING_CLASS = classNames(
    'player-age-gate-warning'
);

const AGE_GATE_WARNING_LABEL_CLASS = classNames(
    'js-age-gate-warning-label'
);

const AGE_GATE_LOCKED_OUT_CLASS = classNames(
    'pl-age-gate-locked-out'
);

const AGE_GATE__ICON = classNames(
    'player-age-gate--icon'
);

const AGE_GATE_NOTIFICATION_CLASS = classNames(
    'player-age-gate-notificiation'
);

const PLAYER_DATEPICKER = classNames(
    'player-datepicker'
);

const AGE_GATE_FORM_CLASS = classNames(
    'player-age-gate-form js-age-gate'
);

const DEFAULT_START_DAY = 1;
const propTypes = {
    handleAgeGatePassed: PropTypes.func.isRequired,
    requiredAge: PropTypes.number.isRequired,
    channelName: PropTypes.string.isRequired,
    t: PropTypes.func.isRequired,
};

// TODO: convert back to PureComponent [VP-3228]
export class AgeGateOverlayComponent extends React.Component {
    constructor(props) {
        super(...arguments);
        const currentYear = new Date().getFullYear();
        const ageGatesFailedCount = this._getAgeGateFailureCount(props.channelName);

        this.state = {
            monthValue: `${MONTH.JANUARY}`,
            dayValue: `${DEFAULT_START_DAY}`,
            yearValue: currentYear,
            ageGatesFailedCount,
        };
        this._handleInputMonthChange = this._handleInputMonthChange.bind(this);
        this._handleInputDayChange = this._handleInputDayChange.bind(this);
        this._handleInputYearChange = this._handleInputYearChange.bind(this);
        this._submitAge = this._submitAge.bind(this);
    }

    render() {
        const ageGateStatus = this._renderAgeGateStatus();

        return (
            <div className={AGE_RESTRICTION_OVERLAY}>
                <div className={PLAYER_CENTER_CONTENT}>
                    <div className={AGE_GATE_CLASS}>
                        {ageGateStatus}
                    </div>
                </div>
            </div>
        );
    }

    _getAgeGateFailuresMap() {
        const ageGateFailure = sessionStore.get(KEY_AGE_GATES_FAILED);
        if (isObject(ageGateFailure)) {
            return ageGateFailure;
        }
        return {};
    }

    _getAgeGateFailureCount(channelName) {
        const ageGateFailure = this._getAgeGateFailuresMap();
        const failureCount = ageGateFailure[channelName];

        if (isFinite(failureCount)) {
            return Math.max(0, failureCount);
        }
        return 0;
    }

    _getSelectedDate() {
        const selectedMonth = Number(this.state.monthValue);
        const selectedDay = Number(this.state.dayValue);
        const selectedYear = Number(this.state.yearValue);

        return {
            month: parseInt(selectedMonth, 10),
            day: parseInt(selectedDay, 10),
            year: parseInt(selectedYear, 10),
        };
    }

    _handleInputMonthChange(event) {
        const selectedMonthValue = Number(event.target.value);
        const maxDays = getMaxDays(selectedMonthValue, Number(this.state.yearValue));

        this.setState({
            monthValue: event.target.value,
        });

        if (this.state.dayValue > maxDays) {
            this.setState({
                dayValue: `${DEFAULT_START_DAY}`,
            });
        }
    }

    _handleInputDayChange(event) {
        this.setState({
            dayValue: event.target.value,
        });
    }

    _handleInputYearChange(event) {
        const selectedMonthValue = Number(this.state.monthValue);
        const yearValue = event.target.value;
        const maxDays = getMaxDays(selectedMonthValue, Number(yearValue));

        this.setState({
            yearValue,
        });

        if (this.state.dayValue > maxDays) {
            this.setState({
                dayValue: `${DEFAULT_START_DAY}`,
            });
        }
    }

    _isOfAgeOrOlder() {
        const selectedMonth = Number(this.state.monthValue);
        const selectedDay = Number(this.state.dayValue);
        const selectedYear = Number(this.state.yearValue);
        const birthDate = new Date(selectedYear, selectedMonth, selectedDay);

        return isMinimumAge(birthDate, this.props.requiredAge);
    }

    _isValidDate() {
        const { month, day, year } = this._getSelectedDate();
        const maxDays = getMaxDays(month, year);

        return day > 0 && day <= maxDays;
    }

    _populateMonthDropdown() {
        const { t } = this.props;
        return (
            <select
                className={MONTH_DATEPICKER_CLASS}
                name="monthValue"
                value={this.state.monthValue}
                onChange={this._handleInputMonthChange}
                onBlur={this._handleInputMonthChange}
            >
                <option className="age-gate-jan" value={MONTH.JANUARY}>{t('January')}</option>
                <option className="age-gate-feb" value={MONTH.FEBRUARY}>{t('February')}</option>
                <option className="age-gate-mar" value={MONTH.MARCH}>{t('March')}</option>
                <option className="age-gate-apr" value={MONTH.APRIL}>{t('April')}</option>
                <option className="age-gate-may" value={MONTH.MAY}>{t('May')}</option>
                <option className="age-gate-jun" value={MONTH.JUNE}>{t('June')}</option>
                <option className="age-gate-jul" value={MONTH.JULY}>{t('July')}</option>
                <option className="age-gate-aug" value={MONTH.AUGUST}>{t('August')}</option>
                <option className="age-gate-sept" value={MONTH.SEPTEMBER}>{t('September')}</option>
                <option className="age-gate-oct" value={MONTH.OCTOBER}>{t('October')}</option>
                <option className="age-gate-nov" value={MONTH.NOVEMBER}>{t('November')}</option>
                <option className="age-gate-dec" value={MONTH.DECEMBER}>{t('December')}</option>
            </select>
        );
    }

    _populateDayDropdown() {
        const { month, year } = this._getSelectedDate();
        const maxDays = getMaxDays(month, year);
        const dayOptions = range(1, maxDays + 1).map(day => {
            return <option key={`day.${day}`} value={day}>{day}</option>;
        });

        return (
            <select
                className={DAY_DATEPICKER_CLASS}
                name="dayValue"
                value={this.state.dayValue}
                onChange={this._handleInputDayChange}
                onBlur={this._handleInputDayChange}
            >
                {dayOptions}
            </select>
        );
    }

    _populateYearDropdown() {
        const currentYear = new Date().getFullYear();
        const endYear = currentYear - 100;
        const yearOptions = range(currentYear, endYear + 1).map(year => {
            return <option key={`year.${year}`} value={year}>{year}</option>;
        });

        return (
            <select
                className={YEAR_DATEPICKER_CLASS}
                name="yearValue"
                value={this.state.yearValue}
                onChange={this._handleInputYearChange}
                onBlur={this._handleInputYearChange}
            >
                {yearOptions}
            </select>
        );
    }

    _setAgeGateFailed() {
        const { channelName } = this.props;
        const ageGatesFailedCount = this._getAgeGateFailureCount(channelName) + 1;
        const updatedAgeGates = assign(this._getAgeGateFailuresMap(), {
            [channelName]: ageGatesFailedCount,
        });

        sessionStore.set(KEY_AGE_GATES_FAILED, updatedAgeGates);
        this.setState({
            ageGatesFailedCount,
        });
    }

    _setAgeGatePassed() {
        this.props.handleAgeGatePassed();
    }

    _submitAge(e) {
        e.preventDefault();
        if (this._isOfAgeOrOlder()) {
            this._setAgeGatePassed();
        } else {
            this._setAgeGateFailed();
        }
    }

    _renderAgeGateNotification() {
        const { t, requiredAge } = this.props;

        if (this.state.ageGatesFailedCount === 1) {
            return (
                <div className={AGE_GATE_FAILED_ONCE_CLASS}>
                    <p>
                        <svg className={AGE_GATE_FAILED_ICON_CLASS}>
                            <use xlinkHref="#age-gate-fail-icon" />
                        </svg>
                    </p>
                    <p className={AGE_GATE_LOCKED_OUT_LABEL_CLASS}>
                        {t('Sorry, you must be over the age of {{age}} to view this content.', { age: requiredAge })}
                    </p>
                </div>
            );
        }
        return this._renderAgeIcon();
    }

    _renderAgeGateStatus() {
        const { t, requiredAge } = this.props;
        const isValidDate = this._isValidDate();
        const selectMonth = this._populateMonthDropdown();
        const selectDay = this._populateDayDropdown();
        const selectYear = this._populateYearDropdown();
        const ageGateNotification = this._renderAgeGateNotification();

        if (this.state.ageGatesFailedCount >= 2) {
            return (
                <div className={AGE_GATE_LOCKED_OUT_CLASS}>
                    <p>
                        <svg className={AGE_GATE_FAILED_ICON_CLASS}>
                            <use xlinkHref="#age-gate-fail-icon" />
                        </svg>
                    </p>

                    <p className={AGE_GATE_LOCKED_OUT_LABEL_CLASS}>
                        {t('Sorry, you must be over the age of {{age}} to view this content.', { age: requiredAge })}
                    </p>
                </div>
            );
        }
        return (
            <div className={AGE_GATE_FORM_CLASS}>
                <div className={AGE_GATE_NOTIFICATION_CLASS}>
                    {ageGateNotification}
                </div>
                <div className={PLAYER_DATEPICKER}>
                    {selectMonth}
                    {selectDay}
                    {selectYear}
                    <Button
                        className={AGE_GATE_SUBMIT_BUTTON_CLASS}
                        onClick={this._submitAge}
                        disabled={!isValidDate}
                    >
                        {AGE_GATE_SUBMIT_LABEL}
                    </Button>
                </div>
            </div>
        );
    }

    _renderAgeIcon() {
        const { t, requiredAge } = this.props;
        const ageGateIcon = requiredAge === 19 ? '#age-gate-icon-19' : '#age-gate-icon-21';

        return (
            <div className={AGE_GATE_WARNING_CLASS}>
                <p>
                    <svg className={AGE_GATE__ICON}>
                        <use xlinkHref={ageGateIcon} />
                    </svg>
                </p>
                <p className={AGE_GATE_WARNING_LABEL_CLASS}>
                    {t('You must be {{age}} to view this content. Please enter your date of birth.',
                        { age: requiredAge })}
                </p>
            </div>
        );
    }
}

AgeGateOverlayComponent.propTypes = propTypes;
export const AgeGateOverlay = translate()(AgeGateOverlayComponent);
