import React from 'react';
import PropTypes from 'prop-types';
import { PlayerCard } from './cards/player-card';
import classNames from 'classnames';
import { getPercentage } from 'util/math';
import { translate } from 'react-i18next';
import { Button } from '../buttons/base-button';

const propTypes = {
    video: PropTypes.shape({
        thumbnailURL: PropTypes.string.isRequired,
        title: PropTypes.string.isRequired,
        channelName: PropTypes.string.isRequired,
        creationDate: PropTypes.string.isRequired,
        duration: PropTypes.number.isRequired,
        id: PropTypes.string.isRequired,
        recommendationType: PropTypes.string.isRequired,
    }).isRequired,
    onClose: PropTypes.func,
    windowObj: PropTypes.object.isRequired,
    onSelect: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
    trackEvent: PropTypes.func.isRequired,
};

const defaultProps = {
    onClose: () => {},
};

export const AUTOPLAY_PERIOD_IN_MS = 20 * 1000;
const TIME_REMAINING_UPDATE_INTERVAL_IN_MS = 100;
const TRACKING_SELECT_FEATURED_VIEW = 'featured';

const PINNED_ITEM_CLASSES = classNames(
    'pl-pinned-item',
    'pl-pinned-item--centered'
);

const PINNED_ITEM_TITLE_CLASSES = classNames(
    'pl-pinned-item__title',
    'pl-flex'
);

const PINNED_ITEM_TITLE_TEXT_CLASSES = classNames(
    'pl-flex__item',
    'pl-flex__item--grow'
);

const CANCEL_BUTTON_CLASSES = classNames(
    'pl-rec__cancel',
    'pl-button',
    'pl-button--hollow',
    'pl-button--white'
);

export class AutoplayingPinnedVODComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            timeRemaining: AUTOPLAY_PERIOD_IN_MS,
        };

        this._updateTimeRemainingIntervalID = null;
        this.handleExitButtonClick = this.handleExitButtonClick.bind(this);
        this.handleSelect = this.handleSelect.bind(this);
    }

    render() {
        const {
            video,
            t,
        } = this.props;

        const {
            timeRemaining,
        } = this.state;

        const timeRemainingText = t('{{timeRemaining}} second', {
            timeRemaining: Math.round(timeRemaining / 1000),
            count: timeRemaining,
        });

        const pinnedItemTitleComponent = (
            <div className={PINNED_ITEM_TITLE_CLASSES}>
                <span className={PINNED_ITEM_TITLE_TEXT_CLASSES}>
                    {t('Playing in')}
                    <span className="pl-pinned-item__title-highlight"> {timeRemainingText}</span>
                </span>
                <Button
                    className={CANCEL_BUTTON_CLASSES}
                    onClick={this.handleExitButtonClick}
                >
                    {t('Cancel')}
                </Button>
            </div>
        );

        const cardInfoComponent = (
            <span>
                <figure className="pl-card__smallicon">
                    <svg>
                        <use xlinkHref="#pl-icon_vod" />
                    </svg>
                </figure>
                <span>{t('Most Recent Video')}</span>
            </span>
        );

        return (
            <div className={PINNED_ITEM_CLASSES}>
                {pinnedItemTitleComponent}
                <PlayerCard
                    className="pl-rec__item"
                    thumbnailURL={video.thumbnailURL}
                    info={cardInfoComponent}
                    onSelect={this.handleSelect}
                    selectItem={video}
                    progressBarPercentage={getPercentage(timeRemaining, AUTOPLAY_PERIOD_IN_MS)}
                    hasThumbZoom={true}
                    hasBorder={true}
                />
            </div>
        );
    }

    componentWillMount() {
        this.props.trackEvent('player_rec_show_featured', {
            /* eslint-disable camelcase */
            recommended_vod_id: this.props.video.id,
            recommended_vod_type: this.props.video.recommendationType,
            /* eslint-enable camelcase */
        });
    }

    componentDidMount() {
        this.startAutoplayCountdown();
    }

    componentWillUnmount() {
        this.stopAutoplayCountdown();
    }

    startAutoplayCountdown() {
        const { windowObj, onSelect, video } = this.props;
        const startTime = windowObj.Date.now();

        this._updateTimeRemainingIntervalID = windowObj.setInterval(() => {
            const currentTime = windowObj.Date.now();
            const timePassed = currentTime - startTime;
            if (timePassed < AUTOPLAY_PERIOD_IN_MS) {
                this.setState({
                    timeRemaining: AUTOPLAY_PERIOD_IN_MS - timePassed,
                });
            } else {
                this.stopAutoplayCountdown();
                this.props.trackEvent('player_rec_autoplay', {
                    /* eslint-disable camelcase */
                    recommended_vod_id: this.props.video.id,
                    recommended_vod_type: this.props.video.recommendationType,
                    /* eslint-enable camelcase */
                });
                onSelect(video);
            }
        }, TIME_REMAINING_UPDATE_INTERVAL_IN_MS);
    }

    stopAutoplayCountdown() {
        const { windowObj } = this.props;

        windowObj.clearInterval(this._updateTimeRemainingIntervalID);
        this._updateTimeRemainingIntervalID = null;
    }

    handleExitButtonClick() {
        this.stopAutoplayCountdown();
        this.props.onClose();
    }

    handleSelect(video) {
        const {
            trackEvent,
            onSelect,
        } = this.props;

        trackEvent('player_rec_select', {
            /* eslint-disable camelcase */
            recommended_vod_id: video.id,
            recommended_vod_type: video.recommendationType,
            recommended_vod_view: TRACKING_SELECT_FEATURED_VIEW,
            /* eslint-enable camelcase */
        });

        this.stopAutoplayCountdown();
        onSelect(video);
    }
}

AutoplayingPinnedVODComponent.defaultProps = defaultProps;
AutoplayingPinnedVODComponent.propTypes = propTypes;
export const AutoplayingPinnedVOD = translate()(AutoplayingPinnedVODComponent);
