import React from 'react';
import PropTypes from 'prop-types';
import { AutoplayProgressBar } from './progress-bar';
import classNames from 'classnames';
import isFinite from 'lodash/isFinite';

const propTypes = {
    thumbnailURL: PropTypes.string.isRequired,
    bread: PropTypes.string,
    title: PropTypes.string,
    info: PropTypes.oneOfType([
        PropTypes.element,
        PropTypes.string,
    ]),
    progressBarPercentage: PropTypes.number,
    onSelect: PropTypes.func,
    selectItem: PropTypes.any,
    className: PropTypes.string,
    hasThumbZoom: PropTypes.bool,
    hasBorder: PropTypes.bool,
};

const defaultProps = {
    thumbnailURL: '',
    hasThumbZoom: false,
    hasBorder: false,
};

/**************************
 *                        *
 *                        *
 * Bread                  *
 * Title                  *
 * Info                   *
 * Autoplay Progress Bar  *
 **************************/
// https://git-aws.internal.justin.tv/pages/web/styles/ui/components/cards.html

export class PlayerCard extends React.Component {
    constructor() {
        super(...arguments);

        this.handleClick = this.handleClick.bind(this);
        this.handleKeyUp = this.handleKeyUp.bind(this);
    }

    render() {
        const {
            thumbnailURL,
            title,
            bread,
            info,
            className,
            progressBarPercentage,
            hasThumbZoom,
            hasBorder,
        } = this.props;

        const PLAYER_CARD_CLASSES = classNames({
            'pl-card': true,
            'pl-card--withprogress': this._isPercentage(progressBarPercentage),
            'pl-card--withborder': hasBorder,
            'pl-card--thumbzoom': hasThumbZoom,
            [className]: true,
        });

        const progressBarComponent = (this._isPercentage(progressBarPercentage)) ? (
            <AutoplayProgressBar
                percentage={progressBarPercentage}
            />
        ) : null;

        return (
            /* Reasons for disabling lint rules:
             * jsx-a11y/no-static-element-interactions: the div must be clickable to be able to select a card
             */
            // eslint-disable-next-line jsx-a11y/no-static-element-interactions
            <div
                className={PLAYER_CARD_CLASSES}
                onClick={this.handleClick}
                onKeyUp={this.handleKeyUp}
                tabIndex={0}
                role="button"
            >
                <a className="pl-card__layout" href="##">
                    <figure className="pl-card__img">
                        <img
                            alt="Video Thumbnail"
                            src={thumbnailURL}
                        />
                    </figure>
                    <div className="pl-card__overlay">
                        <div className="pl-card__body">
                            <h4 className="pl-card__bread ellipsis">
                                {bread}
                            </h4>
                            <h3 className="pl-card__title ellipsis">
                                {title}
                            </h3>
                            <div className="pl-card__info ellipsis">
                                {info}
                            </div>
                        </div>
                        {progressBarComponent}
                    </div>
                </a>
            </div>
        );
    }

    handleKeyUp(event) {
        if (event.key === 'Enter') {
            this.props.onSelect(this.props.selectItem);
        }
    }

    handleClick() {
        this.props.onSelect(this.props.selectItem);
    }

    _isPercentage(percentage) {
        if (
            percentage === undefined ||
            !isFinite(percentage) ||
            percentage < 0 ||
            percentage > 100
        ) {
            return false;
        }

        return true;
    }
}

PlayerCard.defaultProps = defaultProps;
PlayerCard.propTypes = propTypes;
