import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { RecCloseButton } from '../buttons/rec-close-button';
import find from 'lodash/find';
import { PlayerCard } from './cards/player-card';
import { translate } from 'react-i18next';
import { CHANNEL_VODS } from '../../../actions/recommendations';
import { InfoDateLength } from './cards/info-date-length';

// how many columns of VODs will be displayed when the screen is at least `minPlayerWidth` wide
// NOTE: this list is ordered from high->low width requirement! important!
export const RECOMMENDATION_COLUMN_CONFIG = [
    {
        minPlayerWidth: 2381,
        columns: 8,
    },{
        minPlayerWidth: 2084,
        columns: 7,
    }, {
        minPlayerWidth: 1786,
        columns: 6,
    }, {
        minPlayerWidth: 1489,
        columns: 5,
    }, {
        minPlayerWidth: 1191,
        columns: 4,
    }, {
        minPlayerWidth: 893,
        columns: 3,
    }, {
        minPlayerWidth: 596,
        columns: 2,
    }, {
        minPlayerWidth: 0,
        columns: 1,
    },
];

// how many rows of VODs will be displayed when the screen is at least `minPlayerHeight` tall
// NOTE: this list is ordered from high -> low height requirement! important!
export const RECOMMENDATION_ROW_CONFIG = [
    {
        minPlayerHeight: 950,
        rows: 4,
    }, {
        minPlayerHeight: 700,
        rows: 3,
    }, {
        minPlayerHeight: 520,
        rows: 2,
    }, {
        minPlayerHeight: 0,
        rows: 1,
    },
];

const propTypes = {
    height: PropTypes.number.isRequired,
    width: PropTypes.number.isRequired,
    videos: PropTypes.arrayOf(PropTypes.shape({
        thumbnailURL: PropTypes.string.isRequired,
        title: PropTypes.string.isRequired,
        channelName: PropTypes.string.isRequired,
        creationDate: PropTypes.string.isRequired,
        duration: PropTypes.number.isRequired,
        id: PropTypes.string.isRequired,
        recommendationType: PropTypes.string.isRequired,
    }).isRequired).isRequired,
    onSelect: PropTypes.func.isRequired,
    trackEvent: PropTypes.func.isRequired,
    onClose: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
    i18n: PropTypes.shape({
        langCode: PropTypes.string.isRequired,
    }),
    windowObj: PropTypes.object.isRequired,
};

const PANE_CLASSES = classNames(
    'pl-rec',
    'pl-flex',
    'pl-flex--stretch',
    'pl-flex--horizontalCenter'
);

const PANE_TITLE_CLASSES = classNames(
    'pl-rec__title',
    'pl-flex__item'
);

const CARDS_CONTAINER_CLASSES = classNames(
    'pl-rec__container',
    'pl-flex',
    'pl-flex__item',
    'pl-flex--verticalCenter',
    'pl-flex--horizontalCenter'
);

const TRACKING_SELECT_OTHER_VIEW = 'other';

export class OtherVODsComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.handleSelect = this.handleSelect.bind(this);
    }

    componentWillMount() {
        const {
            height,
            width,
            videos,
            trackEvent,
        } = this.props;

        const concatenatedVODIDs = videos.
            slice(0, this._getNumOfCardsVisible(height, width)).
            map(v => v.id).
            join(',');

        trackEvent('player_rec_show_others', {
            /* eslint-disable camelcase */
            recommended_vod_ids: concatenatedVODIDs,
            recommended_vod_types: videos[0].recommendationType,
            /* eslint-enable camelcase */
        });
    }

    render() {
        const {
            videos,
            width,
            height,
            onClose,
            t,
            i18n,
            windowObj,
        } = this.props;

        const numOfCardsVisible = this._getNumOfCardsVisible(height, width);
        const cardComponents = videos.map((video, index) => {
            const PLAYER_CARD_CLASSES = classNames({
                'pl-mg-x-05': true,
                'pl-rec__item': true,
                // eslint-disable-next-line quote-props
                'hidden': (index >= numOfCardsVisible),
            });

            if (video.recommendationType === CHANNEL_VODS) {
                return (
                    <PlayerCard
                        key={`${video.id}_${index >= numOfCardsVisible}`}
                        className={PLAYER_CARD_CLASSES}
                        thumbnailURL={video.thumbnailURL}
                        title={video.title}
                        info={video.game}
                        onSelect={this.handleSelect}
                        selectItem={video}
                        hasThumbZoom={true}
                    />
                );
            }

            const cardInfo = (
                <InfoDateLength
                    length={video.duration}
                    date={new windowObj.Date(video.creationDate)}
                    languageCode={i18n.langCode}
                />
            );

            return (
                <PlayerCard
                    key={`${video.id}_${index >= numOfCardsVisible}`}
                    className={PLAYER_CARD_CLASSES}
                    thumbnailURL={video.thumbnailURL}
                    bread={video.channelName}
                    title={video.title}
                    info={cardInfo}
                    onSelect={this.handleSelect}
                    selectItem={video}
                    hasThumbZoom={true}
                />
            );
        });

        let titleText = null;
        if (videos[0].recommendationType === CHANNEL_VODS) {
            const channelName = videos[0].channelName;
            titleText = t('More from {{channelName}}', { channelName });
        } else {
            titleText = t('Watch Now');
        }

        return (
            <div className={PANE_CLASSES}>
                <RecCloseButton onClick={onClose} />
                <h2 className={PANE_TITLE_CLASSES}>{titleText}</h2>
                <div className={CARDS_CONTAINER_CLASSES}>
                    {cardComponents}
                </div>
            </div>
        );
    }

    _getNumOfCardsVisible(height, width) {
        const { columns } = find(RECOMMENDATION_COLUMN_CONFIG, cfg => width >= cfg.minPlayerWidth);
        const { rows } = find(RECOMMENDATION_ROW_CONFIG, cfg => height >= cfg.minPlayerHeight);

        return columns * rows;
    }

    handleSelect(video) {
        this.props.trackEvent('player_rec_select', {
            /* eslint-disable camelcase */
            recommended_vod_id: video.id,
            recommended_vod_type: video.recommendationType,
            recommended_vod_view: TRACKING_SELECT_OTHER_VIEW,
            /* eslint-enable camelcase */
        });

        this.props.onSelect(video);
    }
}

OtherVODsComponent.propTypes = propTypes;
export const OtherVODs = translate()(OtherVODsComponent);
