import React from 'react';
import PropTypes from 'prop-types';
import { PlayerCard } from './cards/player-card';
import classNames from 'classnames';
import { translate } from 'react-i18next';
import { RecCloseButton } from '../buttons/rec-close-button';
import { FeaturedCollection } from 'ui/containers/collections/featured-collection';

const propTypes = {
    video: PropTypes.shape({
        thumbnailURL: PropTypes.string.isRequired,
        title: PropTypes.string.isRequired,
        channelName: PropTypes.string.isRequired,
        creationDate: PropTypes.string.isRequired,
        duration: PropTypes.number.isRequired,
        id: PropTypes.string.isRequired,
        recommendationType: PropTypes.string.isRequired,
    }).isRequired,
    onClose: PropTypes.func,
    onSelect: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
    trackEvent: PropTypes.func.isRequired,
};

const defaultProps = {
    onClose: () => {},
};

const TRACKING_SELECT_FEATURED_VIEW = 'featured';

const PINNED_ITEM_TITLE_CLASSES = classNames(
    'pl-pinned-item__title',
    'pl-flex'
);

const PINNED_ITEM_TITLE_TEXT_CLASSES = classNames(
    'pl-flex__item',
    'pl-flex__item--grow'
);

export class PinnedVODComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.handleSelect = this.handleSelect.bind(this);
    }

    render() {
        const {
            video,
            t,
            onClose,
        } = this.props;

        const pinnedItemTitleComponent = (
            <div className={PINNED_ITEM_TITLE_CLASSES}>
                <span className={PINNED_ITEM_TITLE_TEXT_CLASSES}>{t('Watch Now')}</span>
                <RecCloseButton onClick={onClose} />
            </div>
        );

        const cardInfoComponent = (
            <span>
                <figure className="pl-card__smallicon">
                    <svg>
                        <use xlinkHref="#pl-icon_vod" />
                    </svg>
                </figure>
                <span>{t('Most Recent Video')}</span>
            </span>
        );

        return (
            <div className="pl-pinned-item">
                {pinnedItemTitleComponent}
                <PlayerCard
                    className="pl-rec__item"
                    thumbnailURL={video.thumbnailURL}
                    info={cardInfoComponent}
                    onSelect={this.handleSelect}
                    selectItem={video}
                    hasThumbZoom={true}
                    hasBorder={true}
                />
                <FeaturedCollection />
            </div>
        );
    }

    componentWillMount() {
        this.props.trackEvent('player_rec_show_featured', {
            /* eslint-disable camelcase */
            recommended_vod_id: this.props.video.id,
            recommended_vod_type: this.props.video.recommendationType,
            /* eslint-enable camelcase */
        });
    }

    handleSelect(video) {
        const {
            trackEvent,
            onSelect,
        } = this.props;

        trackEvent('player_rec_select', {
            /* eslint-disable camelcase */
            recommended_vod_id: video.id,
            recommended_vod_type: video.recommendationType,
            recommended_vod_view: TRACKING_SELECT_FEATURED_VIEW,
            /* eslint-enable camelcase */
        });

        onSelect(video);
    }
}

PinnedVODComponent.defaultProps = defaultProps;
PinnedVODComponent.propTypes = propTypes;
export const PinnedVOD = translate()(PinnedVODComponent);
