import React from 'react';
import PropTypes from 'prop-types';
import * as Timestamp from '../../../util/timestamp';
import { SeekbarPopup } from './seekbar-popup';
import { getPercentage } from '../../../util/math';
import { MARKER_TYPE_GAME_CHANGE, MARKER_TYPE_GAME_METADATA } from '../../../timeline-metadata/manager';

const propTypes = {
    currentTime: PropTypes.number,
    duration: PropTypes.number,
    markers: PropTypes.array,
    onHidingMarkerPreview: PropTypes.func,
    onShowingMarkerPreview: PropTypes.func,
    onMarkerSeek: PropTypes.func,
    seekbarWidth: PropTypes.number,
    trackEvent: PropTypes.func,
    hideThumbnailPreview: PropTypes.bool,
};

const defaultProps = {
    currentTime: 0,
    duration: 0,
    markers: [],
};

// px width of different types of markers
const MARKER_TYPE_WIDTH_MAP = {
    [MARKER_TYPE_GAME_METADATA]: 148,
    [MARKER_TYPE_GAME_CHANGE]: 78,
};

export class SeekbarMarkers extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            showMarkerPopup: false,
            targetDot: null,
        };

        this.handleMouseEnter = this.handleMouseEnter.bind(this);
        this.handleMouseLeave = this.handleMouseLeave.bind(this);
        this.handleMouseDown = this.handleMouseDown.bind(this);
    }

    componentDidMount() {
        this.preloadMarkerSprites();
    }

    render() {
        const { currentTime, duration, markers } = this.props;
        const { showMarkerPopup, targetDot } = this.state;

        if (!this.shouldDisplayMarkers()) {
            return null;
        }

        const markerDots = markers.map((marker, index) => {
            return (
                <button
                    className="player-slider__marker-dot"
                    key={`${marker.title}_${marker.startTime}_${marker.type}`}
                    data-time={marker.startTime}
                    data-marker-index={index}
                    data-passed={marker.startTime <= currentTime}
                    onMouseEnter={this.handleMouseEnter}
                    onMouseLeave={this.handleMouseLeave}
                    onMouseDown={this.handleMouseDown}
                    onKeyDown={this.handleMouseDown}
                    style={{
                        left: `${getPercentage(marker.startTime, duration)}%`,
                    }}
                    tabIndex={0}
                />
            );
        });

        let markerPopup = null;
        if (showMarkerPopup && targetDot) {
            markerPopup = this.getSeekbarPopupForMarker(targetDot);
        }

        return (
            <div>
                {markerPopup}
                <div className="player-slider__marker-dots">
                    {markerDots}
                </div>
            </div>
        );
    }

    handleMouseEnter(e) {
        this.setState({
            showMarkerPopup: true,
            targetDot: e.target,
        });
        this.props.onShowingMarkerPreview();
    }

    handleMouseLeave() {
        this.setState({
            showMarkerPopup: false,
            targetDot: null,
        });
        this.props.onHidingMarkerPreview();
    }

    /* using mousedown instead of click because the slider thumb gets rendered
     * over the marker on mousedown, blocking the click handler from firing
     */
    handleMouseDown(e) {
        const markerDot = e.target;
        const markerTime = Number(markerDot.getAttribute('data-time'));
        const markerIndex = Number(markerDot.getAttribute('data-marker-index'));
        const markerId = this.props.markers[markerIndex].id;

        this.props.onMarkerSeek(markerTime, markerId);
        e.stopPropagation();
    }

    shouldDisplayMarkers() {
        const { duration, markers } = this.props;
        return duration > 0 && markers.length > 0;
    }

    preloadMarkerSprites() {
        /* Creating Images with marker sprite sources makes the browser fetch the image
         * and prime the browser cache for quicker retrieval later.
         */
        this.props.markers.forEach(marker => {
            const image = new Image();
            image.src = marker.thumbnail.imageUrl;
        });
    }

    getSeekbarPopupForMarker(dot) {
        const computedStyles = getComputedStyle(dot);
        const dotLeft = convertStyleValueToFloat(computedStyles, 'left');
        const dotWidth = convertStyleValueToFloat(computedStyles, 'width');
        const dotMarginLeft = convertStyleValueToFloat(computedStyles, 'margin-left');
        const dotPosition = dotLeft + (0.5 * dotWidth) + dotMarginLeft;
        const markerIndex = Number(dot.getAttribute('data-marker-index'));
        const marker = this.props.markers[markerIndex];

        const popupObject = transformToPopupObject(marker);
        return (
            <SeekbarPopup
                popupLeftOffset={dotPosition}
                popupWidth={MARKER_TYPE_WIDTH_MAP[marker.type]}
                popupObject={popupObject}
                seekbarWidth={this.props.seekbarWidth}
                hideThumbnailPreview={this.props.hideThumbnailPreview}
            />
        );
    }
}

function transformToPopupObject(marker) {
    const thumbResizeFactor = MARKER_TYPE_WIDTH_MAP[marker.type] / marker.thumbnail.width;

    return {
        thumbOver: {
            url: marker.thumbnail.imageURL,
            height: marker.thumbnail.height * thumbResizeFactor,
            width: marker.thumbnail.width * thumbResizeFactor,
            sheetWidth: marker.thumbnail.width * marker.thumbnail.cols * thumbResizeFactor,
            x: marker.thumbnail.x * thumbResizeFactor,
            y: marker.thumbnail.y * thumbResizeFactor,
        },
        thumbUnder: {
            url: '',
            height: 0,
            width: 0,
            sheetWidth: 0,
            x: 0,
            y: 0,
        },
        title: marker.title,
        info: marker.info,
        timestamp: Timestamp.toString(marker.startTime, false),
        width: MARKER_TYPE_WIDTH_MAP[marker.type],
    };
}

function convertStyleValueToFloat(computedStyles, style, defaultValue = 0) {
    return isNaN(parseFloat(computedStyles[style])) ? defaultValue : parseFloat(computedStyles[style]);
}

SeekbarMarkers.propTypes = propTypes;
SeekbarMarkers.defaultProps = defaultProps;
