import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { translate } from 'react-i18next';
import { IDENTIFIER_MAIN_MENU } from './menu-manager';
import { Toggle } from '../common/toggle';
import { Button } from '../buttons/base-button';

const propTypes = {
    html5PlayerEnabled: PropTypes.bool.isRequired,
    html5ToggleEnabled: PropTypes.bool.isRequired,
    isShowingVideoStats: PropTypes.bool.isRequired,
    miniPlayerEnabled: PropTypes.bool.isRequired,
    miniPlayerToggleEnabled: PropTypes.bool.isRequired,
    onHTML5Toggle: PropTypes.func,
    onMenuTransition: PropTypes.func,
    onMiniPlayerToggle: PropTypes.func,
    onStatsToggle: PropTypes.func,
    t: PropTypes.func.isRequired,
};

const defaultProps = {
    onHTML5Toggle() {},
    onMenuTransition() {},
    onMiniPlayerToggle() {},
    onStatsToggle() {},
};

const TEXT_HEADER = 'Advanced';
const TEXT_HIDE_VIDEO_STATS = 'Hide Video Stats';
const TEXT_HTML5_PLAYER = 'HTML5 Player';
const TEXT_MINIPLAYER = 'Mini Player';
const TEXT_SHOW_VIDEO_STATS = 'Show Video Stats';

const CLASSES_HEADER_BUTTON = classNames({
    ellipsis: true,
    'qa-header-button': true,
});

const CLASSES_HEADER_DIV = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret-left': true,
});

const CLASSES_SECTION_DIV = classNames({
    'pl-menu__section': true,
    'pl-menu__section--with-sep': true,
});

const CLASSES_ITEM = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'qa-item': true,
});

const CLASSES_SHOW_STATS_BUTTON = classNames({
    ellipsis: true,
    'qa-show-stats-button': true,
});

export class AdvancedMenuComponent extends React.Component {
    constructor() {
        super(...arguments);

        this.handleMenuTransition = this.handleMenuTransition.bind(this);
    }

    handleMenuTransition() {
        this.props.onMenuTransition(IDENTIFIER_MAIN_MENU);
    }

    render() {
        const {
            html5PlayerEnabled,
            html5ToggleEnabled,
            miniPlayerEnabled,
            miniPlayerToggleEnabled,
            isShowingVideoStats,
            onHTML5Toggle,
            onMiniPlayerToggle,
            onStatsToggle,
            t,
        } = this.props;

        const html5Toggle = html5ToggleEnabled ?
            (
                <div className={CLASSES_ITEM}>
                    <Toggle
                        className="qa-html5-toggle"
                        label={TEXT_HTML5_PLAYER}
                        onToggle={onHTML5Toggle}
                        toggledOn={html5PlayerEnabled}
                    />
                </div>
            ) : null;

        const miniPlayerToggle = miniPlayerToggleEnabled ?
            (
                <div className={CLASSES_ITEM}>
                    <Toggle
                        className="qa-miniplayer-toggle"
                        label={TEXT_MINIPLAYER}
                        onToggle={onMiniPlayerToggle}
                        toggledOn={miniPlayerEnabled}
                    />
                </div>
            ) : null;

        return (
            <div className="pl-menu__inner">
                <div className={CLASSES_HEADER_DIV}>
                    <Button
                        className={CLASSES_HEADER_BUTTON}
                        onClick={this.handleMenuTransition}
                    >
                        <span>{t(TEXT_HEADER)}</span>
                    </Button>
                </div>
                <div className={CLASSES_SECTION_DIV}>
                    {html5Toggle}
                    {miniPlayerToggle}
                    <div className={CLASSES_ITEM}>
                        <Button
                            className={CLASSES_SHOW_STATS_BUTTON}
                            onClick={onStatsToggle}
                        >
                            <span>
                                {isShowingVideoStats ? t(TEXT_HIDE_VIDEO_STATS) : t(TEXT_SHOW_VIDEO_STATS)}
                            </span>
                        </Button>
                    </div>
                </div>
            </div>
        );
    }
}

AdvancedMenuComponent.propTypes = propTypes;
AdvancedMenuComponent.defaultProps = defaultProps;

export const AdvancedMenu = translate()(AdvancedMenuComponent);
