import React from 'react';
import PropTypes from 'prop-types';
import CopyToClipboard from 'react-copy-to-clipboard';
import { toString as timestampString } from 'util/timestamp';
import classNames from 'classnames';
import { translate } from 'react-i18next';
import { IDENTIFIER_QUALITY_MENU, IDENTIFIER_ADVANCED_MENU,
         IDENTIFIER_REPORT_ISSUES_MENU, IDENTIFIER_PLAYBACK_SPEED_MENU,
         IDENTIFIER_STAFF_MENU, IDENTIFIER_CLIP_MODERATION_MENU } from './menu-manager';
import { Button } from '../buttons/base-button';
import { TEXT_SPEED } from 'ui/components/settings/playback-speed-menu';
import { PLAYER_CLIPS_EMBED, PLAYER_CLIPS_VIEWING } from 'util/player-type';

export const COPY_URL_FEEDBACK_DELAY = 2000;

const propTypes = {
    playerType: PropTypes.string.isRequired,
    canChangePlaybackSpeed: PropTypes.bool,
    canPopoutPlayer: PropTypes.bool,
    captionsAvailable: PropTypes.bool,
    currentSpeed: PropTypes.number,
    currentTime: PropTypes.number,
    isStaffUser: PropTypes.bool,
    isVod: PropTypes.bool.isRequired,
    isClip: PropTypes.bool.isRequired,
    onMenuTransition: PropTypes.func,
    onPopoutPlayer: PropTypes.func,
    onShowCaptionsOptions: PropTypes.func,
    selectedQualityName: PropTypes.string.isRequired,
    showClipsShortcutOverlay: PropTypes.func.isRequired,
    shouldShowQualityMenu: PropTypes.bool,
    t: PropTypes.func.isRequired,
    vodUrlWithTimestamp: PropTypes.string,
    windowObj: PropTypes.object.isRequired,
};

const defaultProps = {
    canChangePlaybackSpeed: false,
    canPopoutPlayer: true,
    captionsAvailable: false,
    currentSpeed: 1.0,
    currentTime: 0,
    isStaffUser: false,
    onMenuTransition() {},
    onPopoutPlayer() {},
    onShowCaptionsOptions() {},
    shouldShowQualityMenu: true,
    vodUrlWithTimestamp: '',
};

const TEXT_ADVANCED = 'Advanced';
const TEXT_CAPTIONS_OPTIONS = 'Closed Captions Options';
const TEXT_POPOUT = 'Popout Player';
const TEXT_QUALITY = 'Quality';
const TEXT_REPORT_ISSUE = 'Report Playback Issue';
const TEXT_COPY_URL = 'Copy Video URL at {{time}}';
const TEXT_COPIED = 'Copied!';
const TEXT_STAFF = 'Staff';
const TEXT_CLIPS_KEYBOARD_SHORTCUTS = 'View Keyboard Shortcuts';
const TEXT_CLIPS_REPORT_BUTTON = 'Report Clip';
const TEXT_CLIPS_MODERATION_BUTTON = 'Moderation';

const CLASSES_ITEM = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
});

const CLASSES_ITEM_WITH_CARET = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret': true,
});

const CLASSES_ADVANCED_BUTTON = classNames({
    ellipsis: true,
    'qa-advanced-button': true,
});

const CLASSES_REPORT_ISSUE_BUTTON = classNames({
    ellipsis: true,
    'qa-report-issue-button': true,
});

const CLASSES_POPOUT_PLAYER_BUTTON = classNames({
    ellipsis: true,
    'qa-popout-player-button': true,
});

const CLASSES_CAPTIONS_OPTIONS_BUTTON = classNames({
    ellipsis: true,
    'qa-captions-options-button': true,
});

export class MainMenuComponent extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            hasRecentlyCopied: false,
        };

        this.copyUrlTimeoutId = 0;

        this.handleAdvancedClick = this.handleAdvancedClick.bind(this);
        this.handlePlaybackSpeedClick = this.handlePlaybackSpeedClick.bind(this);
        this.handleQualityClick = this.handleQualityClick.bind(this);
        this.handleReportIssueClick = this.handleReportIssueClick.bind(this);
        this.handleStaffClick = this.handleStaffClick.bind(this);
        this.handleUrlCopied = this.handleUrlCopied.bind(this);
        this.handleClipModerationClick = this.handleClipModerationClick.bind(this);
    }

    componentWillUnmount() {
        if (this.copyUrlTimeoutId !== 0) {
            this.props.windowObj.clearTimeout(this.copyUrlTimeoutId);
        }
    }

    handleClipModerationClick() {
        this.props.onMenuTransition(IDENTIFIER_CLIP_MODERATION_MENU);
    }

    handleQualityClick() {
        this.props.onMenuTransition(IDENTIFIER_QUALITY_MENU);
    }

    handlePlaybackSpeedClick() {
        this.props.onMenuTransition(IDENTIFIER_PLAYBACK_SPEED_MENU);
    }

    handleAdvancedClick() {
        this.props.onMenuTransition(IDENTIFIER_ADVANCED_MENU);
    }

    handleReportIssueClick() {
        this.props.onMenuTransition(IDENTIFIER_REPORT_ISSUES_MENU);
    }

    handleStaffClick() {
        this.props.onMenuTransition(IDENTIFIER_STAFF_MENU);
    }

    handleUrlCopied() {
        this.setState({
            hasRecentlyCopied: true,
        });

        this.copyUrlTimeoutId = this.props.windowObj.setTimeout(() => {
            this.setState({
                hasRecentlyCopied: false,
            });
            this.copyUrlTimeoutId = 0;
        }, COPY_URL_FEEDBACK_DELAY);
    }

    renderClipsEmbedMenu() {
        const qualityButton = this._renderQualityButton();
        const playbackSpeedButton = this._renderPlaybackSpeedButton();
        const clipsKeyboardShortcutsButton = this._renderClipsKeyboardShortcutsButton();

        return (
            <div className="pl-menu__inner">
                {qualityButton}
                {playbackSpeedButton}
                <div className="pl-menu__section pl-menu__section--with-sep">
                    {clipsKeyboardShortcutsButton}
                </div>
            </div>
        );
    }

    renderClipsViewingMenu() {
        const qualityButton = this._renderQualityButton();
        const playbackSpeedButton = this._renderPlaybackSpeedButton();
        const clipsKeyboardShortcutsButton = this._renderClipsKeyboardShortcutsButton();
        const moderationButton = this._renderClipModerationButton();
        const reportClipButton = this._renderReportClipButton();

        return (
            <div className="pl-menu__inner">
                {qualityButton}
                {playbackSpeedButton}
                {moderationButton}
                {reportClipButton}
                <div className="pl-menu__section pl-menu__section--with-sep">
                    {clipsKeyboardShortcutsButton}
                </div>
            </div>
        );
    }

    renderStandardMenu() {
        const { shouldShowQualityMenu } = this.props;
        const qualityButton = this._renderQualityButton();
        const playbackSpeedButton = this._renderPlaybackSpeedButton();
        const captionsOptionsButton = this._renderCaptionsOptionsButton();
        const advancedMenuButton = this._renderAdvancedMenuButton();
        const reportIssueMenuButton = this._renderReportIssuesMenuButton();
        const staffMenuButton = this._renderStaffMenuButton();
        const popoutPlayerButton = this._renderPopoutPlayerButton();
        const copyVideoURLButton = this._renderCopyVideoUrlButton();

        const bottomSectionClasses = classNames({
            'pl-menu__section': true,
            'pl-menu__section--with-sep': shouldShowQualityMenu,
        });

        return (
            <div className="pl-menu__inner">
                {qualityButton}
                <div className={bottomSectionClasses}>
                    {playbackSpeedButton}
                    {captionsOptionsButton}
                    {advancedMenuButton}
                    {reportIssueMenuButton}
                    {staffMenuButton}
                    {popoutPlayerButton}
                    {copyVideoURLButton}
                </div>
            </div>
        );
    }

    render() {
        const { playerType } = this.props;

        if (playerType === PLAYER_CLIPS_EMBED) {
            return this.renderClipsEmbedMenu();
        } else if (playerType === PLAYER_CLIPS_VIEWING) {
            return this.renderClipsViewingMenu();
        }

        return this.renderStandardMenu();
    }

    _renderReportClipButton() {
        const { t } = this.props;

        return (
            <div className={CLASSES_ITEM}>
                <Button
                    // onClick={} // TODO: hook up to postMessage
                    className="qa-report-clip"
                >
                    <span>{t(TEXT_CLIPS_REPORT_BUTTON)}</span>
                </Button>
            </div>
        );
    }

    _renderClipsKeyboardShortcutsButton() {
        const { showClipsShortcutOverlay, t } = this.props;

        return (
            <div className={CLASSES_ITEM}>
                <Button
                    onClick={showClipsShortcutOverlay}
                    className="qa-clips-keyboard-shortcuts"
                >
                    <span>{t(TEXT_CLIPS_KEYBOARD_SHORTCUTS)}</span>
                </Button>
            </div>
        );
    }

    _renderClipModerationButton() {
        const { isStaffUser, t } = this.props;

        if (isStaffUser) {
            return (
                <div className={CLASSES_ITEM_WITH_CARET}>
                    <Button
                        onClick={this.handleClipModerationClick}
                        className="qa-clip-moderation-button"
                    >
                        <span>{t(TEXT_CLIPS_MODERATION_BUTTON)}</span>
                    </Button>
                </div>
            );
        }
        return null;
    }

    _renderQualityButton() {
        const { selectedQualityName, shouldShowQualityMenu, t } = this.props;

        if (shouldShowQualityMenu) {
            return (
                <div className={CLASSES_ITEM_WITH_CARET}>
                    <Button
                        onClick={this.handleQualityClick}
                        className="qa-quality-button"
                    >
                        <span>{t(TEXT_QUALITY)}</span>
                        <span className="pl-pill pl-mg-l-05 qa-quality-pill">
                            {selectedQualityName}
                        </span>
                    </Button>
                </div>
            );
        }
        return null;
    }

    _renderPlaybackSpeedButton() {
        const { currentSpeed, isVod, isClip, canChangePlaybackSpeed, t } = this.props;
        const playbackSpeedPillLabel = `${currentSpeed}x`;

        return (isVod || isClip) && canChangePlaybackSpeed ? (
            <div className={CLASSES_ITEM_WITH_CARET}>
                <Button
                    onClick={this.handlePlaybackSpeedClick}
                    className="qa-playback-speed-button"
                >
                    <span>{t(TEXT_SPEED)}</span>
                    <span className="pl-pill pl-mg-l-05 qa-playback-speed-pill">
                        {playbackSpeedPillLabel}
                    </span>
                </Button>
            </div>
        ) : null;
    }

    _renderCaptionsOptionsButton() {
        const { captionsAvailable, onShowCaptionsOptions, t } = this.props;

        if (captionsAvailable) {
            return (
                <div className={CLASSES_ITEM}>
                    <Button
                        className={CLASSES_CAPTIONS_OPTIONS_BUTTON}
                        onClick={onShowCaptionsOptions}
                    >
                        <span>{t(TEXT_CAPTIONS_OPTIONS)}</span>
                    </Button>
                </div>
            );
        }
        return null;
    }

    _renderAdvancedMenuButton() {
        const { t } = this.props;

        return (
            <div className={CLASSES_ITEM_WITH_CARET}>
                <Button
                    className={CLASSES_ADVANCED_BUTTON}
                    onClick={this.handleAdvancedClick}
                >
                    <span>{t(TEXT_ADVANCED)}</span>
                </Button>
            </div>
        );
    }

    _renderReportIssuesMenuButton() {
        const { t } = this.props;

        return (
            <div className={CLASSES_ITEM_WITH_CARET}>
                <Button
                    className={CLASSES_REPORT_ISSUE_BUTTON}
                    onClick={this.handleReportIssueClick}
                >
                    <span>{t(TEXT_REPORT_ISSUE)}</span>
                </Button>
            </div>
        );
    }

    _renderStaffMenuButton() {
        const { isStaffUser } = this.props;

        if (isStaffUser) {
            return (
                <div className={CLASSES_ITEM_WITH_CARET}>
                    <Button
                        className="ellipsis qa-staff-button"
                        onClick={this.handleStaffClick}
                    >
                        <span>{TEXT_STAFF}</span>
                    </Button>
                </div>
            );
        }
        return null;
    }

    _renderPopoutPlayerButton() {
        const { canPopoutPlayer, onPopoutPlayer, t } = this.props;

        if (canPopoutPlayer) {
            return (
                <div className={CLASSES_ITEM}>
                    <Button
                        className={CLASSES_POPOUT_PLAYER_BUTTON}
                        onClick={onPopoutPlayer}
                    >
                        <span>{t(TEXT_POPOUT)}</span>
                    </Button>
                </div>
            );
        }
        return null;
    }

    _renderCopyUrlButton() {
        const { currentTime, vodUrlWithTimestamp, t } = this.props;
        const copyUrlText = t(TEXT_COPY_URL, {
            time: timestampString(currentTime),
        });

        if (this.state.hasRecentlyCopied) {
            return (
                <Button>
                    <span>{t(TEXT_COPIED)}</span>
                </Button>
            );
        }
        return (
            <CopyToClipboard
                text={vodUrlWithTimestamp}
                onSelect={this.handleUrlCopied}
                className="qa-copy-to-clipboard-button"
            >
                <Button>
                    <span>{copyUrlText}</span>
                </Button>
            </CopyToClipboard>
        );
    }

    _renderCopyVideoUrlButton() {
        const { isVod } = this.props;
        const copyUrlButton = this._renderCopyUrlButton();
        const copyUrlButtonClass = classNames({
            [CLASSES_ITEM]: true,
            'pl-menu__item--active': this.state.hasRecentlyCopied,
        });

        if (isVod) {
            return (
                <div className={copyUrlButtonClass}>
                    {copyUrlButton}
                </div>
            );
        }
        return null;
    }
}

MainMenuComponent.propTypes = propTypes;
MainMenuComponent.defaultProps = defaultProps;

export const MainMenu = translate()(MainMenuComponent);
