import React from 'react';
import PropTypes from 'prop-types';
import includes from 'lodash/includes';
import { MainMenu } from 'ui/containers/settings/main-menu';
import { QualityMenu } from 'ui/containers/settings/quality-menu';
import { AdvancedMenu } from 'ui/containers/settings/advanced-menu';
import { ReportIssuesMenu } from 'ui/containers/settings/report-issues-menu';
import { PlaybackSpeedMenu } from 'ui/containers/settings/playback-speed-menu';
import { StaffMenu } from 'ui/containers/settings/staff-menu';
import { ClipsModerationMenu } from 'ui/components/settings/clips-moderation-menu';

export const IDENTIFIER_MAIN_MENU = 'main menu';
export const IDENTIFIER_QUALITY_MENU = 'quality menu';
export const IDENTIFIER_ADVANCED_MENU = 'advanced menu';
export const IDENTIFIER_REPORT_ISSUES_MENU = 'report issues menu';
export const IDENTIFIER_PLAYBACK_SPEED_MENU = 'playback speed menu';
export const IDENTIFIER_STAFF_MENU = 'staff menu';
export const IDENTIFIER_CLIP_MODERATION_MENU = 'clip moderation menu';

export const SETTINGS_INNER_PADDING = 85;

const IDENTIFIERS_ALL = [
    IDENTIFIER_MAIN_MENU,
    IDENTIFIER_QUALITY_MENU,
    IDENTIFIER_ADVANCED_MENU,
    IDENTIFIER_REPORT_ISSUES_MENU,
    IDENTIFIER_PLAYBACK_SPEED_MENU,
    IDENTIFIER_STAFF_MENU,
    IDENTIFIER_CLIP_MODERATION_MENU,
];

const propTypes = {
    playerHeight: PropTypes.number.isRequired,
};

export class MenuManager extends React.Component {
    constructor() {
        super(...arguments);

        this.state = {
            currentMenu: IDENTIFIER_MAIN_MENU,
        };

        this.handleMenuTransition = this.handleMenuTransition.bind(this);
    }

    handleMenuTransition(identifier) {
        if (includes(IDENTIFIERS_ALL, identifier)) {
            this.setState({
                currentMenu: identifier,
            });
        }
    }

    render() {
        const { playerHeight } = this.props;
        const currentMenu = this._renderCurrentMenu();
        const settingsStyle = {
            maxHeight: `${playerHeight - SETTINGS_INNER_PADDING}px`,
        };

        return (
            <div
                className="pl-menu"
                style={settingsStyle}
            >
                {currentMenu}
            </div>
        );
    }

    // eslint-disable-next-line complexity
    _renderCurrentMenu() {
        switch (this.state.currentMenu) {
        case IDENTIFIER_QUALITY_MENU:
            return (
                <QualityMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        case IDENTIFIER_ADVANCED_MENU:
            return (
                <AdvancedMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        case IDENTIFIER_REPORT_ISSUES_MENU:
            return (
                <ReportIssuesMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        case IDENTIFIER_PLAYBACK_SPEED_MENU:
            return (
                <PlaybackSpeedMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        case IDENTIFIER_STAFF_MENU:
            return (
                <StaffMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        case IDENTIFIER_CLIP_MODERATION_MENU:
            return (
                <ClipsModerationMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        case IDENTIFIER_MAIN_MENU:
        default:
            return (
                <MainMenu
                    onMenuTransition={this.handleMenuTransition}
                />
            );
        }
    }
}

MenuManager.propTypes = propTypes;
