import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { translate } from 'react-i18next';
import { IDENTIFIER_MAIN_MENU } from './menu-manager';
import { PlaybackSpeedButton } from 'ui/components/settings/playback-speed-button';

const propTypes = {
    currentSpeed: PropTypes.number.isRequired,
    onMenuTransition: PropTypes.func,
    onSelectPlaybackSpeed: PropTypes.func,
    t: PropTypes.func.isRequired,
};

const defaultProps = {
    onMenuTransition() {},
    onSelectPlaybackSpeed() {},
};

export const PLAYBACK_SPEEDS = Object.freeze([
    0.25,
    0.5,
    0.75,
    1.0,
    1.25,
    1.5,
    2.0,
]);

export const TEXT_SPEED = 'Speed';
export const TEXT_NORMAL = 'Normal';

const CLASSES_MENU_HEADER = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret-left': true,
});

const CLASSES_SECTION_DIV = classNames({
    'pl-menu__section': true,
    'pl-menu__section--with-sep': true,
});

export class PlaybackSpeedMenuComponent extends React.Component {
    constructor() {
        super(...arguments);

        this.handleMenuTransition = this.handleMenuTransition.bind(this);
    }

    handleMenuTransition() {
        this.props.onMenuTransition(IDENTIFIER_MAIN_MENU);
    }

    render() {
        const { t } = this.props;
        const speedButtons = this._getSpeedSelectionButtons();

        return (
            <div className="pl-menu__inner">
                <div className={CLASSES_MENU_HEADER}>
                    <button
                        className="qa-header-button"
                        onClick={this.handleMenuTransition}
                    >
                        {t(TEXT_SPEED)}
                    </button>
                </div>
                <div className={CLASSES_SECTION_DIV}>
                    {speedButtons}
                </div>
            </div>
        );
    }

    _getSpeedSelectionButtons() {
        const { currentSpeed, onSelectPlaybackSpeed, t } = this.props;
        return PLAYBACK_SPEEDS.map(speed => {
            const isActive = currentSpeed === speed;
            const label = speed === 1 ? `1x (${t(TEXT_NORMAL)})` : `${speed}x`;

            return (
                <PlaybackSpeedButton
                    isActive={isActive}
                    key={speed}
                    label={label}
                    onClick={onSelectPlaybackSpeed}
                    speedValue={speed}
                />
            );
        });
    }
}

PlaybackSpeedMenuComponent.propTypes = propTypes;
PlaybackSpeedMenuComponent.defaultProps = defaultProps;

export const PlaybackSpeedMenu = translate()(PlaybackSpeedMenuComponent);
