import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { translate } from 'react-i18next';
import { IDENTIFIER_MAIN_MENU } from './menu-manager';

const propTypes = {
    t: PropTypes.func.isRequired,
    onReportIssueSubmit: PropTypes.func.isRequired,
    onMenuTransition: PropTypes.func,
    windowObj: PropTypes.object.isRequired,
};

const defaultProps = {
    onMenuTransition() {},
};

export const REPORT_FEEDBACK_DELAY = 2000;

/* eslint-disable quotes */
const TEXT_HEADER = 'Report Playback Issue';
const TEXT_DROPDOWN_HEADER = 'Noticed a video playback issue? Let us know!';
const TEXT_SUBMIT = 'Submit';
const TEXT_THANK_YOU = 'Thanks for your report';
const TEXT_SELECT = 'Select';
const TEXT_STUTTER_BOTH = 'Audio and video stutter';
const TEXT_STUTTER_VIDEO = 'Video stutters, but audio is fine';
const TEXT_BLACK_SCREEN = `Video is completely black or doesn't load`;
const TEXT_AV_DESYNC = `Audio and video aren't synced`;
const TEXT_FULLSCREEN_BROKEN = `Fullscreen playback doesn't work`;
const TEXT_AD_VOLUME = `Advertisement can't be muted or is too loud`;
const TEXT_AD_REPEAT = 'Advertisement has played too many times';
/* eslint-enable quotes */

const CLASSES_SELECT_ELEMENT = classNames({
    // eslint-disable-next-line camelcase
    'pl-form__input': true,
    ellipsis: true,
});

const CLASSES_HEADER_DIV = classNames({
    'pl-menu__item': true,
    'pl-menu__item--block': true,
    'pl-menu__item--with-caret-left': true,
});

const CLASSES_HEADER_BUTTON = classNames({
    ellipsis: true,
    'qa-header-button': true,
});

const CLASSES_SECTION_DIV = classNames({
    'pl-menu__section': true,
    'pl-menu__section--with-sep': true,
});

const CLASSES_BUTTON_CONTAINER = classNames({
    'pl-flex': true,
    'pl-flex--horizontalEnd': true,
    'pl-pd-r-1': true,
    'pl-pd-t-1': true,
    'pl-pd-b-1': true,
});

const CLASSES_SUBMIT_BUTTON = classNames({
    'pl-button': true,
    'pl-button--hollow': true,
    'qa-submit-button': true,
});

export class ReportIssuesMenuComponent extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            value: '',
            hasRecentlySubmitted: false,
        };

        this.handleMenuTransition = this.handleMenuTransition.bind(this);
        this.handleIssueSelected = this.handleIssueSelected.bind(this);
        this.handleSubmit = this.handleSubmit.bind(this);
    }

    handleMenuTransition() {
        this.props.onMenuTransition(IDENTIFIER_MAIN_MENU);
    }

    handleIssueSelected(e) {
        this.setState({
            value: e.target.value,
        });
    }

    handleSubmit(e) {
        e.preventDefault();

        if (this.state.value === '') {
            return;
        }

        this.props.onReportIssueSubmit(this.state.value);
        this.setState({
            hasRecentlySubmitted: true,
        });
        this.props.windowObj.setTimeout(() => {
            this.setState({
                hasRecentlySubmitted: false,
            });
        }, REPORT_FEEDBACK_DELAY);
    }

    render() {
        const { t } = this.props;
        const feedbackForm = this._renderFeedbackForm();

        return (
            <div className="pl-menu__inner">
                <div className={CLASSES_HEADER_DIV}>
                    <button
                        className={CLASSES_HEADER_BUTTON}
                        onClick={this.handleMenuTransition}
                    >
                        {t(TEXT_HEADER)}
                    </button>
                </div>
                <div className={CLASSES_SECTION_DIV}>
                    {feedbackForm}
                </div>
            </div>
        );
    }

    _createSelectElement() {
        const { t } = this.props;

        return (
            <select
                className={CLASSES_SELECT_ELEMENT}
                onBlur={this.handleIssueSelected}
                defaultValue={''}
            >
                <option value="" disabled={true}>{t(TEXT_SELECT)}</option>
                <option value="stutter-both">{t(TEXT_STUTTER_BOTH)}</option>
                <option value="stutter-video">{t(TEXT_STUTTER_VIDEO)}</option>
                <option value="black-screen">{t(TEXT_BLACK_SCREEN)}</option>
                <option value="av-desync">{t(TEXT_AV_DESYNC)}</option>
                <option value="fullscreen-broken">{t(TEXT_FULLSCREEN_BROKEN)}</option>
                <option value="ad-volume">{t(TEXT_AD_VOLUME)}</option>
                <option value="ad-repeat">{t(TEXT_AD_REPEAT)}</option>
            </select>
        );
    }

    _renderFeedbackForm() {
        const { t } = this.props;
        const selectElement = this._createSelectElement();
        const { hasRecentlySubmitted } = this.state;

        if (hasRecentlySubmitted) {
            return (<span>{t(TEXT_THANK_YOU)}</span>);
        }
        return (
            <form
                onSubmit={this.handleSubmit}
            >
                <div className="form__group">
                    <p>{t(TEXT_DROPDOWN_HEADER)}</p>
                    {selectElement}
                </div>
                <div className={CLASSES_BUTTON_CONTAINER}>
                    <button
                        type="submit"
                        className={CLASSES_SUBMIT_BUTTON}
                    >
                        {t(TEXT_SUBMIT)}
                    </button>
                </div>
            </form>
        );
    }
}

ReportIssuesMenuComponent.propTypes = propTypes;
ReportIssuesMenuComponent.defaultProps = defaultProps;

export const ReportIssuesMenu = translate()(ReportIssuesMenuComponent);
