import React from 'react';
import PropTypes from 'prop-types';
import whatInput from 'what-input';
import { SliderLeft } from './slider/slider-left';
import { SliderThumb } from './slider/slider-thumb';
import find from 'lodash/find';

const propTypes = {
    afterSliderChildren: PropTypes.oneOfType([
        PropTypes.arrayOf(PropTypes.element),
        PropTypes.element,
    ]),
    behindSliderChildren: PropTypes.oneOfType([
        PropTypes.arrayOf(PropTypes.element),
        PropTypes.element,
    ]),
    classNames: PropTypes.shape({
        slider: PropTypes.string,
        sliderLeft: PropTypes.string,
        sliderThumb: PropTypes.string,
    }),
    dragHandlers: PropTypes.shape({
        onStart: PropTypes.func,
        onDrag: PropTypes.func,
        onStop: PropTypes.func,
    }),
    getOptimizedValue: PropTypes.func,
    max: PropTypes.number.isRequired,
    min: PropTypes.number.isRequired,
    onBlur: PropTypes.func,
    onClick: PropTypes.func,
    onKeyDown: PropTypes.func,
    onMouseMove: PropTypes.func,
    onMouseOut: PropTypes.func,
    sliderTabIndex: PropTypes.number,
    windowObj: PropTypes.object,
    value: PropTypes.number,
    valueOptimizationEnabled: PropTypes.bool,
};

const defaultProps = {
    value: 0,
    classNames: {},
    sliderTabIndex: 0,
    valueOptimizationEnabled: false,
    getOptimizedValue: () => 0,
};

export class Slider extends React.Component {
    constructor() {
        super(...arguments);
        this.state = {
            isDragging: false,
            'data-whatInput': '',
            currentDragValue: 0,
            touchIdentifier: -1,
            optimizedValue: 0,
            requestAnimationID: 0,
        };
        this.handleTouchStart = this.handleTouchStart.bind(this);
        this.handleTouchEnd = this.handleTouchEnd.bind(this);
        this.handleMouseDown = this.handleMouseDown.bind(this);
        this.handleMouseUp = this.handleMouseUp.bind(this);
        this.handleClick = this.handleClick.bind(this);
        this.handleFocus = this.handleFocus.bind(this);
        this._sliderRefHandler = this._sliderRefHandler.bind(this);
        this.updateOptimizedValue = this.updateOptimizedValue.bind(this);
    }

    componentDidMount() {
        if (this.props.valueOptimizationEnabled) {
            this.updateOptimizedValue();
        }
    }

    componentWillUnmount() {
        if (this.props.valueOptimizationEnabled) {
            this.props.windowObj.cancelAnimationFrame(this.state.requestAnimationID);
        }
    }

    updateOptimizedValue() {
        this.setState({
            optimizedValue: this.props.getOptimizedValue(),
            requestAnimationID: this.props.windowObj.requestAnimationFrame(this.updateOptimizedValue),
        });
    }

    render() {
        const {
            afterSliderChildren,
            behindSliderChildren,
            classNames,
            max,
            min,
            onBlur,
            onKeyDown,
            onMouseMove,
            onMouseOut,
            sliderTabIndex,
            value,
            valueOptimizationEnabled,
        } = this.props;

        let sliderValue = 0;

        if (this.state.isDragging) {
            sliderValue = this.state.currentDragValue;
        } else if (valueOptimizationEnabled) {
            sliderValue = this.state.optimizedValue;
        } else {
            sliderValue = value;
        }

        const leftSlider = (
            <SliderLeft
                className={classNames.sliderLeft}
                max={max}
                min={min}
                value={sliderValue}
            />
        );

        const draggableThumb = (
            <SliderThumb
                className={classNames.sliderThumb}
                max={max}
                min={min}
                onTouchStart={this.handleTouchStart}
                onTouchEnd={this.handleTouchEnd}
                onMouseDown={this.handleMouseDown}
                onMouseUp={this.handleMouseUp}
                value={sliderValue}
            />
        );

        return (
            /* Reasons for disabling lint rules:
             * jsx-a11y/no-static-element-interactions: the div must be clickable to be able to seek to a point
             */
            // eslint-disable-next-line jsx-a11y/no-static-element-interactions
            <div
                aria-valuemax={max}
                aria-valuemin={min}
                aria-valuenow={sliderValue}
                className={classNames.slider}
                data-isDragging={this.state.isDragging}
                data-whatInput={this.state['data-whatInput']}
                onBlur={onBlur}
                onClick={this.handleClick}
                onFocus={this.handleFocus}
                onKeyDown={onKeyDown}
                onMouseMove={onMouseMove}
                onMouseOut={onMouseOut}
                onMouseDown={this.handleMouseDown}
                onMouseUp={this.handleMouseUp}
                onTouchStart={this.handleTouchStart}
                onTouchEnd={this.handleTouchEnd}
                ref={this._sliderRefHandler}
                role="slider"
                tabIndex={sliderTabIndex}
            >
                {behindSliderChildren}
                {leftSlider}
                {afterSliderChildren}
                {draggableThumb}
            </div>
        );
    }

    _sliderRefHandler(slider) {
        this.$slider = slider;
    }

    calculateValueAtPosition(pageX = 0) {
        const { max, min } = this.props;
        const { left, width } = this.$slider.getBoundingClientRect();

        if (width === 0) {
            return 0;
        }

        // Bound the percentage to a value between 0 and 1
        const percentageOfSlider = Math.min(1, Math.max(0, (pageX - left) / width));

        return percentageOfSlider * (max - min) + min;
    }

    handleClick(e) {
        const value = this.calculateValueAtPosition(e.pageX);

        const handler = this.props.onClick;
        if (handler) {
            handler(value, e);
        }
    }

    onDragStart(e) {
        const value = this.calculateValueAtPosition(e.pageX);

        this.setState({
            isDragging: true,
            currentDragValue: value,
        });

        const handler = this.props.dragHandlers.onStart;
        if (handler) {
            handler(value, e);
        }
    }

    onDragEvent(e) {
        const value = this.calculateValueAtPosition(e.pageX);

        this.setState({
            currentDragValue: value,
        });

        const handler = this.props.dragHandlers.onDrag;
        if (handler) {
            handler(value, e);
        }
    }

    onDragEnd(e) {
        this.setState({
            isDragging: false,
            currentDragValue: 0,
        });

        const value = this.calculateValueAtPosition(e.pageX);

        const handler = this.props.dragHandlers.onStop;
        if (handler) {
            handler(value, e);
        }
    }

    handleFocus() {
        this.setState({
            'data-whatInput': whatInput.ask(),
        });
    }

    addMoveHandlers(isTouch = false) {
        if (!this.$slider) {
            return;
        }

        const { ownerDocument } = this.$slider;

        if (isTouch) {
            ownerDocument.addEventListener('touchmove', this);
            ownerDocument.addEventListener('touchend', this);
            ownerDocument.addEventListener('touchcancel', this);
        } else {
            ownerDocument.addEventListener('mousemove', this);
            ownerDocument.addEventListener('mouseup', this);
        }
    }

    removeMoveHandlers(isTouch = false) {
        if (!this.$slider) {
            return;
        }

        const { ownerDocument } = this.$slider;

        if (isTouch) {
            ownerDocument.removeEventListener('touchmove', this);
            ownerDocument.removeEventListener('touchend', this);
            ownerDocument.removeEventListener('touchcancel', this);
        } else {
            ownerDocument.removeEventListener('mousemove', this);
            ownerDocument.removeEventListener('mouseup', this);
        }
    }

    handleTouchStart(e) {
        const firstTouchEvent = e.touches[0];
        this.setState({
            touchIdentifier: firstTouchEvent.identifier,
        });

        this.onDragStart(firstTouchEvent);
        this.addMoveHandlers(true);
    }

    handleTouchMove(e) {
        const startingTouch = find(e.touches, touch => {
            return touch.identifier === this.state.touchIdentifier;
        });

        if (startingTouch) {
            this.onDragEvent(startingTouch);
        }
    }

    handleTouchEnd(e) {
        const startingTouch = find(e.changedTouches, touch => {
            return touch.identifier === this.state.touchIdentifier;
        });

        if (startingTouch) {
            this.onDragEnd(startingTouch);
            this.removeMoveHandlers(true);
            this.setState({
                touchIdentifier: -1,
            });
        }
    }

    handleMouseDown(e) {
        this.onDragStart(e);
        this.addMoveHandlers(false);
    }

    handleMouseMove(e) {
        this.onDragEvent(e);
    }

    handleMouseUp(e) {
        this.onDragEnd(e);
        this.removeMoveHandlers(false);
    }

    // eslint-disable-next-line complexity
    handleEvent(e) {
        switch (e.type) {
        case 'touchmove':
            return this.handleTouchMove(e);
        case 'touchend':
        case 'touchcancel':
            return this.handleTouchEnd(e);
        case 'mousemove':
            return this.handleMouseMove(e);
        case 'mouseup':
            return this.handleMouseUp(e);
        default:
            throw Error(`unexpected Slider event type: ${e.type}`);
        }
    }
}

Slider.propTypes = propTypes;
Slider.defaultProps = defaultProps;
