import React from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { Stat } from './stat';
import { Button } from 'ui/components/buttons/base-button';

const propTypes = {
    stats: PropTypes.shape({
        bufferSize: PropTypes.number,
        displayResolution: PropTypes.string,
        skippedFrames: PropTypes.number,
        fps: PropTypes.number,
        hlsLatencyBroadcaster: PropTypes.number,
        hlsLatencyEncoder: PropTypes.number,
        memoryUsage: PropTypes.string,
        playbackRate: PropTypes.number,
        videoResolution: PropTypes.string,
        backendVersion: PropTypes.string,
    }).isRequired,
    isLive: PropTypes.bool.isRequired,
    onClose: PropTypes.func.isRequired,
};

const CLOSE_BUTTON_CLASS = classNames(
    'player-button',
    'player-button--noscale',
    'player-button--close'
);

export function Stats({ stats, isLive, onClose }) {
    let latencyToBroadcaster = null;
    let latencyToEncoder = null;

    if (isLive) {
        latencyToBroadcaster = (
            <Stat
                statName={'Latency To Broadcaster'}
                value={`${stats.hlsLatencyBroadcaster} sec.`}
            />
        );
        latencyToEncoder = (
            <Stat
                statName={'Latency To Encoder'}
                value={`${stats.hlsLatencyEncoder} sec.`}
            />
        );
    }

    return (
        <ul className="pl-playback-stats js-playback-stats">
            <Button
                className={CLOSE_BUTTON_CLASS}
                onClick={onClose}
            >
                <svg>
                    <use xlinkHref="#icon_close" />
                </svg>
            </Button>
            <Stat
                statName={'Video Resolution'}
                value={stats.videoResolution}
            />
            <Stat
                statName={'Display Resolution'}
                value={stats.displayResolution}
            />
            <Stat
                statName={'FPS'}
                value={stats.fps}
            />
            <Stat
                statName={'Skipped Frames'}
                value={stats.skippedFrames}
            />
            <Stat
                statName={'Buffer Size'}
                value={`${stats.bufferSize} sec.`}
            />
            { latencyToBroadcaster }
            { latencyToEncoder }
            <Stat
                statName={'Playback Rate'}
                value={`${stats.playbackRate} Kbps`}
            />
            <Stat
                statName={'Memory Usage'}
                value={stats.memoryUsage}
            />
            <Stat
                statName={'Backend Version'}
                value={stats.backendVersion}
            />
        </ul>
    );
}

Stats.propTypes = propTypes;
