import React from 'react';
import PropTypes from 'prop-types';
import { translate, Interpolate } from 'react-i18next';
import classNames from 'classnames';

const propTypes = {
    gameName: PropTypes.string,
    gameUrl: PropTypes.string,
    lang: PropTypes.object,
    live: PropTypes.bool,
    isMobile: PropTypes.bool,
    name: PropTypes.string,
    onClick: PropTypes.func,
    streamUrl: PropTypes.string,
    title: PropTypes.string,
    viewerCount: PropTypes.number,
    className: PropTypes.string,
};

const GAME_INFO_CLASS = classNames(
    'player-text-link',
    'player-text-link--no-color',
    'qa-game-name'
);

const NAME_CLASSES = {
    container: 'player-streaminfo__name',
    a: classNames(
        'player-text-link',
        'player-text-link--no-color',
        'qa-display-name'
    ),
};

export const StreamInfoDisplayComponent = ({
    gameName,
    gameUrl,
    lang,
    live,
    name,
    streamUrl,
    onClick,
    title,
    viewerCount,
    className,
    isMobile,
}) => {
    const localeViewerCount = Number(viewerCount).toLocaleString(lang.shortCode);
    const gameInfoComponent = (
        <a
            className={GAME_INFO_CLASS}
            href={gameUrl}
            target="_blank"
        >
            {gameName}
        </a>
    );
    const viewerCountComponent = (
        <span>
            {localeViewerCount}
        </span>
    );
    let liveDataComponent = null;

    const TITLE_CLASSES = classNames({
        'player-streaminfo__title': true,
        'qa-stream-title': true,
        'player-streaminfo__title--hide': live && isMobile,
    });

    if (live) {
        liveDataComponent = (
            <div className="player-streaminfo__viewers">
                <Interpolate
                    i18nKey={'playing {{game}} for {{viewerCount}} viewer'}
                    game={gameInfoComponent}
                    viewerCount={viewerCountComponent}
                    options={{
                        count: viewerCount,
                    }}
                />
            </div>
        );
    }

    return (
        <div className={className}>
            <div className={NAME_CLASSES.container}>
                <a
                    className={NAME_CLASSES.a}
                    href={streamUrl}
                    target="_blank"
                    onClick={onClick}
                >
                    {name}
                </a>
            </div>
            <div className={TITLE_CLASSES}>
                {title}
            </div>
            <div className="player-streaminfo__game">
                {liveDataComponent}
            </div>
        </div>
    );
};

StreamInfoDisplayComponent.propTypes = propTypes;
export const StreamInfoDisplay = translate()(StreamInfoDisplayComponent);
