import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { BackgroundBanner as BackgroundBannerComponent } from 'ui/components/background-banner';
import { OFFLINE_STATUS } from 'state/online-status';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_NONE } from 'stream/null';
import { AdContentTypes } from 'actions/ads';
import { contentIsShowing } from 'actions/playback';

const propTypes = {
    bannerType: PropTypes.string.isRequired,
    imgUrl: PropTypes.string,
    onContentShowing: PropTypes.func.isRequired,
};

const OFFLINE_BANNER = 'offline-banner';
const VIDEO_PREVIEW = 'video-preview';
const NONE = 'none';

export const mapStateToProps = state => {
    const { channel, preview } = state.streamMetadata;
    const bannerType = getBannerType(state);
    const offlineBannerSrc = channel.videoBanner;
    const videoPreviewSrc = preview.template;

    if (bannerType === NONE) {
        return {
            bannerType,
            imgUrl: '',
        };
    }

    return {
        bannerType,
        imgUrl: bannerType === OFFLINE_BANNER ?
            createBGImageUrl(offlineBannerSrc) :
            createBGImageUrl(videoPreviewSrc),
    };
};

export const mapDispatchToProps = dispatch => ({
    onContentShowing() {
        dispatch(contentIsShowing());
    },
});

export class BackgroundBannerContainer extends React.Component {
    componentWillReceiveProps({ bannerType }) {
        if (bannerType !== NONE) {
            this.props.onContentShowing();
        }
    }

    render() {
        const {
            bannerType,
            imgUrl,
        } = this.props;

        if (bannerType === NONE) {
            return null;
        }

        return (
            <BackgroundBannerComponent
                imgUrl={imgUrl}
                bgType={bannerType}
            />
        );
    }
}

BackgroundBannerContainer.propTypes = propTypes;
export const BackgroundBanner = connect(
    mapStateToProps,
    mapDispatchToProps
)(BackgroundBannerContainer);

/**
 * Determines which banner to show - none, offline banner, video preview
 *
 * @param {Object} - takes a subset of state
 * @return {String}
 */
function getBannerType({ playback, stream, onlineStatus, ads }) {
    const adPlaying = ads.currentMetadata.contentType !== AdContentTypes.NONE;
    const isOffline = onlineStatus === OFFLINE_STATUS;
    const { autoplay, ended, hasPlayed, paused } = playback;
    const { contentType } = stream;

    const showVideoPreview = (!autoplay && !hasPlayed && paused && !isOffline) ||
        contentType === CONTENT_MODE_NONE;

    const showOfflineBanner = contentType === CONTENT_MODE_LIVE &&
        !adPlaying &&
        isOffline &&
        ended;

    if (showVideoPreview) {
        return VIDEO_PREVIEW;
    }

    if (showOfflineBanner) {
        return OFFLINE_BANNER;
    }

    return NONE;
}

/**
 * Returns a image url with the default width and height
 *
 * @param {urlString} BG image url - https://static-cdn.jtvnw.net/previews-ttv/live_user_monstercat-{width}x{height}.jpg
 * @return {String}
 */
export function createBGImageUrl(str) {
    const VOD_BACKGROUND_RESOLUTION = '320x240';
    const TEMPLATE_RESOLUTION = '{width}x{height}';
    const DEFAULT_BG_RESOLUTION = '1280x720';

    return str.
    replace(TEMPLATE_RESOLUTION, DEFAULT_BG_RESOLUTION).
    replace(`-${VOD_BACKGROUND_RESOLUTION}`, `-${DEFAULT_BG_RESOLUTION}`);
}
