import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { Button } from 'ui/components/buttons/base-button';
import classNames from 'classnames';
import { translate } from 'react-i18next';

const propTypes = {
    channelName: PropTypes.string.isRequired,
    isOwnChannel: PropTypes.bool.isRequired,
    isFollowing: PropTypes.bool.isRequired,
    onFollowChannel: PropTypes.func.isRequired,
    onUnfollowChannel: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
};

export const mapStateToProps = state => ({
    channelName: state.streamMetadata.channel.name,
    isOwnChannel: state.streamMetadata.channel.id === state.user.id,
    isFollowing: state.follow.following,
});

export function FollowButtonComponent(props) {
    const {
        channelName,
        isFollowing,
        onFollowChannel,
        onUnfollowChannel,
        t,
    } = props;

    if (props.isOwnChannel) {
        return null;
    }

    const ACTIVE_BUTTON_CLASS = classNames(
        'pl-button',
        'pl-button--follow',
        'pl-button--icon',
        'pl-button--status'
    );

    const DEFAULT_BUTTON_CLASS = classNames(
        'pl-button',
        'pl-button--follow',
        'pl-button--icon'
    );

    const unfollowButton = (
        <Button
            className={ACTIVE_BUTTON_CLASS}
            onClick={onUnfollowChannel}
        >
            <span className="pl-button__status pl-button__status--hover">
                <figure className="pl-icon">
                    <svg>
                        <use xlinkHref="#icon_unheart" />
                    </svg>
                </figure>
                <span className="pl-button__text">{t('Unfollow')}</span>
            </span>
            <span className="pl-button__status pl-button__status--primary">
                <figure className="pl-icon">
                    <svg>
                        <use xlinkHref="#icon_heart" />
                    </svg>
                </figure>
                <span className="pl-button__text">{t('Following')}</span>
            </span>
        </Button>
    );

    const followButton = (
        <Button
            className={DEFAULT_BUTTON_CLASS}
            onClick={onFollowChannel}
        >
            <figure className="pl-icon">
                <svg>
                    <use xlinkHref="#icon_heart" />
                </svg>
            </figure>
            <span className="pl-button__text">
                {t('Follow {{channelName}} on Twitch', { channelName })}
            </span>
        </Button>
    );

    return isFollowing ? unfollowButton : followButton;
}

FollowButtonComponent.propTypes = propTypes;

const withStoreProps = connect(mapStateToProps);
export const FollowButton = translate()(withStoreProps(FollowButtonComponent));
