import React from 'react';
import PropTypes from 'prop-types';
import includes from 'lodash/includes';
import { connect } from 'react-redux';
import { PausePlayButton as PausePlayButtonComponent } from 'ui/components/buttons/pause-play-button';
import { pause, play, mutePlayer } from 'actions/video-api';
import { trackEvent } from 'actions/analytics-tracker';
import { CLIPS_PLAYER_TYPES } from 'util/player-type';

const propTypes = {
    pause: PropTypes.func.isRequired,
    play: PropTypes.func.isRequired,
    trackPlayPause: PropTypes.func.isRequired,
    unmutePlayer: PropTypes.func.isRequired,
    paused: PropTypes.bool.isRequired,
    showReplay: PropTypes.bool.isRequired,
    isMobileWeb: PropTypes.bool.isRequired,
};

export const mapStateToProps = ({ playback, env }) => ({
    paused: playback.paused,
    showReplay: (
        includes(CLIPS_PLAYER_TYPES, env.playerType) &&
        playback.paused &&
        playback.ended
    ),
    isMobileWeb: env.isMobileLocation,
});

export const mapDispatchToProps = dispatch => ({
    play() {
        dispatch(play());
    },

    pause() {
        dispatch(pause());
    },

    trackPlayPause() {
        dispatch(trackEvent('player_click_playpause'));
    },

    unmutePlayer() {
        dispatch(mutePlayer(false));
    },
});

export class PausePlayButtonContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.onPauseHandler = this.onPauseHandler.bind(this);
        this.onPlayHandler = this.onPlayHandler.bind(this);
    }

    render() {
        const { paused, showReplay } = this.props;

        return (
            <PausePlayButtonComponent
                onPause={this.onPauseHandler}
                onPlay={this.onPlayHandler}
                paused={paused}
                showReplay={showReplay}
            />
        );
    }

    onPlayHandler() {
        const { play, trackPlayPause } = this.props;

        play();
        trackPlayPause();
    }

    onPauseHandler() {
        const {
            pause,
            trackPlayPause,
            unmutePlayer,
            isMobileWeb,
        } = this.props;

        pause();
        trackPlayPause();

        if (isMobileWeb) {
            unmutePlayer();
        }
    }
}

PausePlayButtonContainer.propTypes = propTypes;

export const PausePlayButton = connect(mapStateToProps, mapDispatchToProps)(PausePlayButtonContainer);
