import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { WatchTwitchButton as WatchTwitchButtonComponent } from 'ui/components/buttons/watch-twitch-button';
import { pause } from 'actions/video-api';
import { channelUrl, videoUrl } from 'api';
import { twitchHost } from 'settings';
import { toURLString } from 'util/timestamp';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_NONE } from 'stream/null';
import { CONTENT_MODE_CLIP } from 'stream/clip';
import {
    TRACKING_MEDIUM_EMBED,
    TRACKING_PROPERTY_TWITCH_LOGO,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import { ttAnalyticsQuery } from 'ui/player-types/clips/utils/link-helper';
import { translate } from 'react-i18next';

const propTypes = {
    currentTime: PropTypes.number.isRequired,
    streamType: PropTypes.string,
    channelName: PropTypes.string.isRequired,
    videoId: PropTypes.string,
    metadataUrl: PropTypes.string,
    windowObj: PropTypes.object.isRequired,
    showBranding: PropTypes.bool.isRequired,
    pausePlayer: PropTypes.func.isRequired,
    t: PropTypes.func.isRequired,
};

const defaultProps = {
    streamType: CONTENT_MODE_NONE,
    videoId: '',
};

export const mapStateToProps = ({ playback, stream, streamMetadata, window, playerOptions }) => ({
    currentTime: playback.currentTime,
    streamType: stream.contentType,
    channelName: streamMetadata.channel.name,
    videoId: streamMetadata.videoId,
    metadataUrl: streamMetadata.url,
    windowObj: window,
    showBranding: playerOptions.branding,
});

export const mapDispatchToProps = dispatch => ({
    pausePlayer() {
        dispatch(pause());
    },
});

export class WatchTwitchButtonContainer extends React.Component {
    constructor() {
        super(...arguments);
        this.handleClick = this.handleClick.bind(this);
        this.getButtonLabel = this.getButtonLabel.bind(this);
    }

    shouldComponentUpdate(nextProps) {
        const hasChanged = propName => nextProps[propName] !== this.props[propName];

        return (
            hasChanged('streamType') ||
            hasChanged('channelName') ||
            hasChanged('videoId') ||
            hasChanged('showBranding') ||
            hasChanged('pausePlayer')
        );
    }

    render() {
        if (!this.props.showBranding) {
            return null;
        }

        return (
            <WatchTwitchButtonComponent
                onClick={this.handleClick}
                buttonLabel={this.getButtonLabel()}
            />
        );
    }

    getButtonLabel() {
        const {
            streamType,
            t,
        } = this.props;

        return streamType === CONTENT_MODE_CLIP ? t('View clip page') : t('Watch on Twitch');
    }

    handleClick() {
        const {
            streamType,
            windowObj,
            pausePlayer,
        } = this.props;

        let url = '';

        switch (streamType) {
        case CONTENT_MODE_LIVE:
            url = this._getLiveURL();
            break;
        case CONTENT_MODE_VOD:
            url = this._getVODURL();
            break;
        case CONTENT_MODE_CLIP:
            url = this._getClipURL();
            break;
        case CONTENT_MODE_NONE:
            url = twitchHost;
            break;
        }

        windowObj.open(url, '_blank');
        pausePlayer();
    }

    _getClipURL() {
        const { metadataUrl } = this.props;

        return ttAnalyticsQuery(metadataUrl, TRACKING_MEDIUM_EMBED, TRACKING_PROPERTY_TWITCH_LOGO);
    }

    _getVODURL() {
        const {
            channelName,
            videoId,
            currentTime,
        } = this.props;

        const timestamp = toURLString(Math.round(currentTime));
        return videoUrl(channelName, videoId, { t: timestamp });
    }

    _getLiveURL() {
        return channelUrl(this.props.channelName);
    }
}

WatchTwitchButtonContainer.propTypes = propTypes;
WatchTwitchButtonContainer.defaultProps = defaultProps;

export const WatchTwitchButton = translate()(connect(mapStateToProps, mapDispatchToProps)(WatchTwitchButtonContainer));
