import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { ClosedCaptions as ClosedCaptionsComponent } from 'ui/components/closed-captions/closed-captions';

const propTypes = {
    enabled: PropTypes.bool.isRequired,
    captionsData: PropTypes.object,
    captionsStyle: PropTypes.shape({
        fontSize: PropTypes.number,
        font: PropTypes.string,
        fontOpacity: PropTypes.string,
        alignment: PropTypes.string,
        edge: PropTypes.string,
        verticalPosition: PropTypes.string,
        backgroundColorName: PropTypes.string,
        backgroundOpacity: PropTypes.string,
        windowColorName: PropTypes.string,
        windowOpacity: PropTypes.string,
        fontUnderline: PropTypes.string,
        fontBold: PropTypes.string,
        fontItalic: PropTypes.string,
    }).isRequired,
};

const NEWLINE_REGEX = /\r?\n|\r/g;

export const mapStateToProps = ({ captions, ui }) => ({
    enabled: captions.enabled && !ui.isMini,
    captionsData: captions.data,
    captionsStyle: captions.style,
});

export function ClosedCaptionsContainer({ enabled, captionsData, captionsStyle }) {
    if (
        !enabled ||
        !captionsData ||
        (Array.isArray(captionsData.data) && captionsData.data.length < 1)
    ) {
        return null;
    }

    const captionsText = parseCaptions(captionsData);

    return (
        <ClosedCaptionsComponent
            captionsText={captionsText}
            captionsStyle={captionsStyle}
        />
    );
}

ClosedCaptionsContainer.propTypes = propTypes;
export const ClosedCaptions = connect(mapStateToProps)(ClosedCaptionsContainer);

/**
 * Takes raw data from mediaplayer and parses it into an array of strings
 *
 * @param {Object} data, text - data is an Array of characters with row/col positions
 *                              text is a string with \n characters
 * @return {Array} Array of strings, each item being a row of captions.
 */
export function parseCaptions({ data, text }) {
    const str = (text && text.length > 0) ? text.trim() : '';

    if (str) {
        return generateCaptionsFromString(str);
    }

    if (Array.isArray(data)) {
        return generateCaptionsFromArray(data);
    }
}

function generateCaptionsFromArray(captionsData) {
    return captionsData.
        reduce((accumulator, caption) => {
            const { char, col } = caption;
            if (col === 0) {
                accumulator.push('');
            }

            const [firstCharacter] = char;
            /* eslint no-param-reassign: ["error", { "props": false }] */
            accumulator[accumulator.length - 1] += firstCharacter ? firstCharacter : '';

            return accumulator;
        }, []);
}

function generateCaptionsFromString(text) {
    return String(text).
        split(NEWLINE_REGEX);
}
