import React from 'react';
import PropTypes from 'prop-types';
import enhanceWithClickOutside from 'react-click-outside';
import TransitionGroup from 'react-transition-group/TransitionGroup';
import findIndex from 'lodash/findIndex';
import { connect } from 'react-redux';
import { SIDEBAR_VIEW } from 'state/collection';
import { AnimatedCollectionSidebar } from 'ui/components/collections/collection-sidebar';
import { closeCollectionSidebar } from 'actions/collection';
import { selectCollectionVideo } from 'actions/playback';
import { ADVERTISEMENT_SCREEN, STITCHED_ADVERTISEMENT_SCREEN } from 'actions/screen';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';

const propTypes = {
    collection: PropTypes.object.isRequired,
    showSidebar: PropTypes.bool.isRequired,
    currentVideoId: PropTypes.string.isRequired,
    onCloseSidebar: PropTypes.func.isRequired,
    onSelectCollectionItem: PropTypes.func.isRequired,
    windowObj: PropTypes.object.isRequired,
    shouldRender: PropTypes.bool.isRequired,
};

export const mapStateToProps = ({ collection, stream, window, ui, screen }) => ({
    collection,
    showSidebar: collection.currentView === SIDEBAR_VIEW,
    currentVideoId: stream.contentType === CONTENT_MODE_VOD ? stream.videoId : '',
    windowObj: window,
    shouldRender: (
        !ui.isMini &&
        screen[0] !== STITCHED_ADVERTISEMENT_SCREEN &&
        screen[0] !== ADVERTISEMENT_SCREEN &&
        collection.id !== '' &&
        stream.contentType === CONTENT_MODE_VOD
    ),
});

export const mapDispatchToProps = dispatch => ({
    onCloseSidebar() {
        dispatch(closeCollectionSidebar());
    },
    onSelectCollectionItem(collectionItemId, collectionId) {
        dispatch(selectCollectionVideo(`v${collectionItemId}`, collectionId));
    },
});

const CLOSE_DELAY_TIME = 400; // ms

export class CollectionSideBarContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.handleSelectItem = this.handleSelectItem.bind(this);
        this.beginClosingSidebar = this.beginClosingSidebar.bind(this);
        this.stopClosingSidebar = this.stopClosingSidebar.bind(this);
        this.closeSidebarTimeout = null;
    }

    render() {
        const {
            collection,
            currentVideoId,
            showSidebar,
            onCloseSidebar,
            shouldRender,
        } = this.props;

        if (!shouldRender) {
            return null;
        }

        this._preloadThumbnails();

        const currentItemIndex = findIndex(collection.items, item => {
            return `v${item.itemId}` === currentVideoId;
        });

        const animatedCollectionSidebar = (
            <AnimatedCollectionSidebar
                currentItemIndex={currentItemIndex}
                collection={collection}
                onSelectItem={this.handleSelectItem}
                onClose={onCloseSidebar}
                key={`${collection.id}_${currentItemIndex}`}
                onMouseLeave={this.beginClosingSidebar}
                onMouseEnter={this.stopClosingSidebar}
            />
        );

        const sidebar = showSidebar ? animatedCollectionSidebar : null;

        return (
            <TransitionGroup component={'div'}>
                {sidebar}
            </TransitionGroup>
        );
    }

    _preloadThumbnails() {
        this.props.collection.items.forEach(item => {
            const image = new Image();
            image.src = item.thumbnails.small;
        });
    }

    handleSelectItem(collectionItemId) {
        this.props.onSelectCollectionItem(collectionItemId, this.props.collection.id);
    }

    beginClosingSidebar() {
        const { windowObj, onCloseSidebar } = this.props;

        this.closeSidebarTimeout = windowObj.setTimeout(() => onCloseSidebar() , CLOSE_DELAY_TIME);
    }

    stopClosingSidebar() {
        this.props.windowObj.clearTimeout(this.closeSidebarTimeout);
    }

    // Invoked by the enhanceWithClickOutside HOC
    handleClickOutside() {
        if (this.props.showSidebar) {
            this.props.onCloseSidebar();
        }
    }

    componentWillUnmount() {
        this.props.windowObj.clearTimeout(this.closeSidebarTimeout);
    }
}

CollectionSideBarContainer.propTypes = propTypes;
// eslint-disable-next-line max-len
export const CollectionSidebar = connect(mapStateToProps, mapDispatchToProps)(enhanceWithClickOutside(CollectionSideBarContainer));

