import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { COLLAPSED_VIEW } from 'state/collection';
import { openCollectionSidebar } from 'actions/collection';
import { CollectionTopBar as CollectionTopBarComponent } from 'ui/components/collections/collection-topbar';
import { CONTENT_SCREEN } from 'actions/screen';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';

const SIDEBAR_OPEN_HOVER_TIME = 250;

const propTypes = {
    collectionItems: PropTypes.array.isRequired,
    shouldShowTopBar: PropTypes.bool.isRequired,
    showDivider: PropTypes.bool.isRequired,
    stream: PropTypes.object.isRequired,
    showSidebar: PropTypes.func,
    title: PropTypes.string.isRequired,
    win: PropTypes.object.isRequired,
};

const defaultProps = {
    showSidebar() {},
};

export const mapStateToProps = ({ collection, playerOptions, ui, stream, window: win, screenMode, screen }) => ({
    title: collection.title,
    shouldShowTopBar: (
        collection.currentView === COLLAPSED_VIEW &&
        !ui.isMini &&
        screen[0] === CONTENT_SCREEN
    ),
    showDivider: playerOptions.showInfo || screenMode.isTheatreMode,
    collectionItems: collection.items,
    stream,
    win,
});

export const mapDispatchToProps = dispatch => ({
    showSidebar() {
        dispatch(openCollectionSidebar());
    },
});

export class CollectionTopBarContainer extends React.Component {
    constructor() {
        super(...arguments);

        this.showSidebar = this.showSidebar.bind(this);
        this.onMouseEnter = this.onMouseEnter.bind(this);
        this.onMouseLeave = this.onMouseLeave.bind(this);
        this._hoverTimeout = 0;
    }

    onMouseEnter() {
        this._hoverTimeout = this.props.win.setTimeout(this.showSidebar, SIDEBAR_OPEN_HOVER_TIME);
    }

    onMouseLeave() {
        this.props.win.clearTimeout(this._hoverTimeout);
        this._hoverTimeout = 0;
    }

    showSidebar() {
        this.props.showSidebar();
    }

    render() {
        const {
            collectionItems,
            shouldShowTopBar,
            showDivider,
            stream,
            title,
        } = this.props;

        if (!shouldShowTopBar || stream.contentType !== CONTENT_MODE_VOD) {
            return null;
        }

        return (
            <CollectionTopBarComponent
                onMouseEnter={this.onMouseEnter}
                onMouseLeave={this.onMouseLeave}
                collectionItems={collectionItems}
                showDivider={showDivider}
                showSidebar={this.showSidebar}
                streamId={String(stream.videoId).substr(1)}
                title={title}
            />
        );
    }
}

CollectionTopBarContainer.propTypes = propTypes;
CollectionTopBarContainer.defaultProps = defaultProps;

export const CollectionTopBar = connect(mapStateToProps, mapDispatchToProps)(CollectionTopBarContainer);
